"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.boundary = exports.shopifyApp = exports.ApiVersion = exports.BillingInterval = exports.DeliveryMethod = exports.LogSeverity = exports.LATEST_API_VERSION = exports.AppDistribution = exports.LoginErrorType = void 0;
var tslib_1 = require("tslib");
require("@shopify/shopify-api/adapters/web-api");
var shopify_api_1 = require("@shopify/shopify-api");
var runtime_1 = require("@shopify/shopify-api/runtime");
var types_1 = require("./types");
var version_1 = require("./version");
var webhooks_1 = require("./auth/webhooks");
var authenticate_1 = require("./auth/admin/authenticate");
var authenticate_2 = require("./auth/webhooks/authenticate");
var authenticate_3 = require("./auth/public/authenticate");
var override_logger_1 = require("./override-logger");
var helpers_1 = require("./auth/helpers");
var login_1 = require("./auth/login/login");
var headers_1 = require("./boundary/headers");
var error_1 = require("./boundary/error");
var types_2 = require("./types");
Object.defineProperty(exports, "LoginErrorType", { enumerable: true, get: function () { return types_2.LoginErrorType; } });
Object.defineProperty(exports, "AppDistribution", { enumerable: true, get: function () { return types_2.AppDistribution; } });
var shopify_api_2 = require("@shopify/shopify-api");
Object.defineProperty(exports, "LATEST_API_VERSION", { enumerable: true, get: function () { return shopify_api_2.LATEST_API_VERSION; } });
Object.defineProperty(exports, "LogSeverity", { enumerable: true, get: function () { return shopify_api_2.LogSeverity; } });
Object.defineProperty(exports, "DeliveryMethod", { enumerable: true, get: function () { return shopify_api_2.DeliveryMethod; } });
Object.defineProperty(exports, "BillingInterval", { enumerable: true, get: function () { return shopify_api_2.BillingInterval; } });
Object.defineProperty(exports, "ApiVersion", { enumerable: true, get: function () { return shopify_api_2.ApiVersion; } });
(0, runtime_1.setAbstractRuntimeString)(function () {
    return "Remix";
});
/**
 * Creates an object your app will use to interact with Shopify.
 *
 * @param appConfig Configuration options for your shopify app.  For example, the scopes your app needs.
 * @returns `ShopifyApp` An object constructed using your appConfig.  It has methods for interacting with Shopify.
 *
 * @example
 * The minimum viable configuration
 * ```ts
 * import { shopifyApp } from "@shopify/shopify-app-remix";
 *
 * const shopify = shopifyApp({
 *   apiKey: process.env.SHOPIFY_API_KEY!,
 *   apiSecretKey: process.env.SHOPIFY_API_SECRET!,
 *   scopes: process.env.SCOPES?.split(",")!,
 *   appUrl: process.env.SHOPIFY_APP_URL!,
 * });
 * export default shopify;
 * ```
 */
function shopifyApp(appConfig) {
    var api = deriveApi(appConfig);
    var config = deriveConfig(appConfig, api.config);
    var logger = (0, override_logger_1.overrideLogger)(api.logger);
    if (appConfig.webhooks) {
        api.webhooks.addHandlers(appConfig.webhooks);
    }
    var params = { api: api, config: config, logger: logger };
    var oauth = new authenticate_1.AuthStrategy(params);
    var shopify = {
        sessionStorage: config.sessionStorage,
        addDocumentResponseHeaders: (0, helpers_1.addDocumentResponseHeadersFactory)(params),
        registerWebhooks: (0, webhooks_1.registerWebhooksFactory)(params),
        authenticate: {
            admin: oauth.authenticateAdmin.bind(oauth),
            public: (0, authenticate_3.authenticatePublicFactory)(params),
            webhook: (0, authenticate_2.authenticateWebhookFactory)(params),
        },
    };
    if (isAppStoreApp(shopify, appConfig) ||
        isSingleMerchantApp(shopify, appConfig)) {
        shopify.login = (0, login_1.loginFactory)(params);
    }
    return shopify;
}
exports.shopifyApp = shopifyApp;
/**
 * A collection of functions that handle the necessary code for error boundaries in routes using authenticate.admin.
 */
exports.boundary = {
    /**
     * A function that handles errors or thrown responses.
     *
     * @example
     * Catching errors in a route
     * ```ts
     * // app/routes/admin/widgets.ts
     * import { boundary } from "@shopify/shopify-app-remix";
     *
     * export function ErrorBoundary() {
     *   return boundary.error(useRouteError());
     * }
     * ```
     */
    error: error_1.errorBoundary,
    /**
     * A function that sets the appropriate document repsonse headers.
     *
     * @example
     * Catching errors in a route
     * ```ts
     * // app/routes/admin/widgets.ts
     * import { boundary } from "@shopify/shopify-app-remix";
     *
     * export const headers = (headersArgs) => {
     *   return boundary.headers(headersArgs);
     * };
     * ```
     */
    headers: headers_1.headersBoundary,
};
function isAppStoreApp(_shopify, config) {
    return config.distribution === types_1.AppDistribution.AppStore;
}
function isSingleMerchantApp(_shopify, config) {
    return config.distribution === types_1.AppDistribution.SingleMerchant;
}
function deriveApi(appConfig) {
    var _a, _b;
    var appUrl;
    try {
        appUrl = new URL(appConfig.appUrl);
    }
    catch (error) {
        throw new shopify_api_1.ShopifyError('Invalid appUrl provided. Please provide a valid URL.');
    }
    /* eslint-disable no-process-env */
    if (appUrl.hostname === 'localhost' && !appUrl.port && process.env.PORT) {
        appUrl.port = process.env.PORT;
    }
    /* eslint-enable no-process-env */
    appConfig.appUrl = appUrl.origin;
    var userAgentPrefix = "Shopify Remix Library v".concat(version_1.SHOPIFY_REMIX_LIBRARY_VERSION);
    if (appConfig.userAgentPrefix) {
        userAgentPrefix = "".concat(appConfig.userAgentPrefix, " | ").concat(userAgentPrefix);
    }
    var cleanApiConfig = tslib_1.__assign(tslib_1.__assign({}, appConfig), { hostName: appUrl.host, hostScheme: appUrl.protocol.replace(':', ''), userAgentPrefix: userAgentPrefix, isEmbeddedApp: (_a = appConfig.isEmbeddedApp) !== null && _a !== void 0 ? _a : true, apiVersion: (_b = appConfig.apiVersion) !== null && _b !== void 0 ? _b : shopify_api_1.LATEST_API_VERSION, isCustomStoreApp: appConfig.distribution === types_1.AppDistribution.ShopifyAdmin });
    return (0, shopify_api_1.shopifyApi)(cleanApiConfig);
}
function deriveConfig(appConfig, apiConfig) {
    var _a, _b, _c;
    if (!appConfig.sessionStorage) {
        throw new shopify_api_1.ShopifyError('Please provide a valid session storage. See https://github.com/Shopify/shopify-app-js/blob/main/README.md#session-storage-options for options.');
    }
    var authPathPrefix = appConfig.authPathPrefix || '/auth';
    appConfig.distribution = (_a = appConfig.distribution) !== null && _a !== void 0 ? _a : types_1.AppDistribution.AppStore;
    return tslib_1.__assign(tslib_1.__assign(tslib_1.__assign({}, appConfig), apiConfig), { canUseLoginForm: appConfig.distribution !== types_1.AppDistribution.ShopifyAdmin, useOnlineTokens: (_b = appConfig.useOnlineTokens) !== null && _b !== void 0 ? _b : false, hooks: (_c = appConfig.hooks) !== null && _c !== void 0 ? _c : {}, sessionStorage: appConfig.sessionStorage, auth: {
            path: authPathPrefix,
            callbackPath: "".concat(authPathPrefix, "/callback"),
            patchSessionTokenPath: "".concat(authPathPrefix, "/session-token"),
            exitIframePath: "".concat(authPathPrefix, "/exit-iframe"),
            loginPath: "".concat(authPathPrefix, "/login"),
        } });
}
//# sourceMappingURL=index.js.map