'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

require('@shopify/shopify-api/adapters/web-api');
var shopifyApi = require('@shopify/shopify-api');
var runtime = require('@shopify/shopify-api/runtime');
var types = require('./types.js');
var version = require('./version.js');
var register = require('./auth/webhooks/register.js');
var authenticate$2 = require('./auth/admin/authenticate.js');
var authenticate$1 = require('./auth/webhooks/authenticate.js');
var authenticate = require('./auth/public/authenticate.js');
var overrideLogger = require('./override-logger.js');
var addResponseHeaders = require('./auth/helpers/add-response-headers.js');
require('@remix-run/server-runtime');
require('isbot');
var login = require('./auth/login/login.js');
var headers = require('./boundary/headers.js');
var error = require('./boundary/error.js');

runtime.setAbstractRuntimeString(() => {
  return `Remix`;
});

/**
 * Creates an object your app will use to interact with Shopify.
 *
 * @param appConfig Configuration options for your shopify app.  For example, the scopes your app needs.
 * @returns `ShopifyApp` An object constructed using your appConfig.  It has methods for interacting with Shopify.
 *
 * @example
 * The minimum viable configuration
 * ```ts
 * import { shopifyApp } from "@shopify/shopify-app-remix";
 *
 * const shopify = shopifyApp({
 *   apiKey: process.env.SHOPIFY_API_KEY!,
 *   apiSecretKey: process.env.SHOPIFY_API_SECRET!,
 *   scopes: process.env.SCOPES?.split(",")!,
 *   appUrl: process.env.SHOPIFY_APP_URL!,
 * });
 * export default shopify;
 * ```
 */
function shopifyApp(appConfig) {
  const api = deriveApi(appConfig);
  const config = deriveConfig(appConfig, api.config);
  const logger = overrideLogger.overrideLogger(api.logger);
  if (appConfig.webhooks) {
    api.webhooks.addHandlers(appConfig.webhooks);
  }
  const params = {
    api,
    config,
    logger
  };
  const oauth = new authenticate$2.AuthStrategy(params);
  const shopify = {
    sessionStorage: config.sessionStorage,
    addDocumentResponseHeaders: addResponseHeaders.addDocumentResponseHeadersFactory(params),
    registerWebhooks: register.registerWebhooksFactory(params),
    authenticate: {
      admin: oauth.authenticateAdmin.bind(oauth),
      public: authenticate.authenticatePublicFactory(params),
      webhook: authenticate$1.authenticateWebhookFactory(params)
    }
  };
  if (isAppStoreApp(shopify, appConfig) || isSingleMerchantApp(shopify, appConfig)) {
    shopify.login = login.loginFactory(params);
  }
  return shopify;
}

/**
 * A collection of functions that handle the necessary code for error boundaries in routes using authenticate.admin.
 */
const boundary = {
  /**
   * A function that handles errors or thrown responses.
   *
   * @example
   * Catching errors in a route
   * ```ts
   * // app/routes/admin/widgets.ts
   * import { boundary } from "@shopify/shopify-app-remix";
   *
   * export function ErrorBoundary() {
   *   return boundary.error(useRouteError());
   * }
   * ```
   */
  error: error.errorBoundary,
  /**
   * A function that sets the appropriate document repsonse headers.
   *
   * @example
   * Catching errors in a route
   * ```ts
   * // app/routes/admin/widgets.ts
   * import { boundary } from "@shopify/shopify-app-remix";
   *
   * export const headers = (headersArgs) => {
   *   return boundary.headers(headersArgs);
   * };
   * ```
   */
  headers: headers.headersBoundary
};
function isAppStoreApp(_shopify, config) {
  return config.distribution === types.AppDistribution.AppStore;
}
function isSingleMerchantApp(_shopify, config) {
  return config.distribution === types.AppDistribution.SingleMerchant;
}
function deriveApi(appConfig) {
  var _appConfig$isEmbedded, _appConfig$apiVersion;
  let appUrl;
  try {
    appUrl = new URL(appConfig.appUrl);
  } catch (error) {
    throw new shopifyApi.ShopifyError('Invalid appUrl provided. Please provide a valid URL.');
  }

  /* eslint-disable no-process-env */
  if (appUrl.hostname === 'localhost' && !appUrl.port && process.env.PORT) {
    appUrl.port = process.env.PORT;
  }
  /* eslint-enable no-process-env */
  appConfig.appUrl = appUrl.origin;
  let userAgentPrefix = `Shopify Remix Library v${version.SHOPIFY_REMIX_LIBRARY_VERSION}`;
  if (appConfig.userAgentPrefix) {
    userAgentPrefix = `${appConfig.userAgentPrefix} | ${userAgentPrefix}`;
  }
  const cleanApiConfig = {
    ...appConfig,
    hostName: appUrl.host,
    hostScheme: appUrl.protocol.replace(':', ''),
    userAgentPrefix,
    isEmbeddedApp: (_appConfig$isEmbedded = appConfig.isEmbeddedApp) !== null && _appConfig$isEmbedded !== void 0 ? _appConfig$isEmbedded : true,
    apiVersion: (_appConfig$apiVersion = appConfig.apiVersion) !== null && _appConfig$apiVersion !== void 0 ? _appConfig$apiVersion : shopifyApi.LATEST_API_VERSION,
    isCustomStoreApp: appConfig.distribution === types.AppDistribution.ShopifyAdmin
  };
  return shopifyApi.shopifyApi(cleanApiConfig);
}
function deriveConfig(appConfig, apiConfig) {
  var _appConfig$distributi, _appConfig$useOnlineT, _appConfig$hooks;
  if (!appConfig.sessionStorage) {
    throw new shopifyApi.ShopifyError('Please provide a valid session storage. See https://github.com/Shopify/shopify-app-js/blob/main/README.md#session-storage-options for options.');
  }
  const authPathPrefix = appConfig.authPathPrefix || '/auth';
  appConfig.distribution = (_appConfig$distributi = appConfig.distribution) !== null && _appConfig$distributi !== void 0 ? _appConfig$distributi : types.AppDistribution.AppStore;
  return {
    ...appConfig,
    ...apiConfig,
    canUseLoginForm: appConfig.distribution !== types.AppDistribution.ShopifyAdmin,
    useOnlineTokens: (_appConfig$useOnlineT = appConfig.useOnlineTokens) !== null && _appConfig$useOnlineT !== void 0 ? _appConfig$useOnlineT : false,
    hooks: (_appConfig$hooks = appConfig.hooks) !== null && _appConfig$hooks !== void 0 ? _appConfig$hooks : {},
    sessionStorage: appConfig.sessionStorage,
    auth: {
      path: authPathPrefix,
      callbackPath: `${authPathPrefix}/callback`,
      patchSessionTokenPath: `${authPathPrefix}/session-token`,
      exitIframePath: `${authPathPrefix}/exit-iframe`,
      loginPath: `${authPathPrefix}/login`
    }
  };
}

Object.defineProperty(exports, 'ApiVersion', {
  enumerable: true,
  get: function () { return shopifyApi.ApiVersion; }
});
Object.defineProperty(exports, 'BillingInterval', {
  enumerable: true,
  get: function () { return shopifyApi.BillingInterval; }
});
Object.defineProperty(exports, 'DeliveryMethod', {
  enumerable: true,
  get: function () { return shopifyApi.DeliveryMethod; }
});
Object.defineProperty(exports, 'LATEST_API_VERSION', {
  enumerable: true,
  get: function () { return shopifyApi.LATEST_API_VERSION; }
});
Object.defineProperty(exports, 'LogSeverity', {
  enumerable: true,
  get: function () { return shopifyApi.LogSeverity; }
});
exports.AppDistribution = types.AppDistribution;
exports.LoginErrorType = types.LoginErrorType;
exports.boundary = boundary;
exports.shopifyApp = shopifyApp;
