import React from 'react';
import { memoize } from '@shopify/function-enhancers';
import { pseudotranslate } from '@shopify/i18n';
import { MissingTranslationError, MissingReplacementError } from '../errors.mjs';
import { DEFAULT_FORMAT } from './interpolate.mjs';

const MISSING_TRANSLATION = Symbol('Missing translation');
const CARDINAL_PLURALIZATION_KEY_NAME = 'count';
const ORDINAL_PLURALIZATION_KEY_NAME = 'ordinal';
const SEPARATOR = '.';
const UNICODE_NUMBERING_SYSTEM = '-u-nu-';
const LATIN = 'latn';

const isString = value => typeof value === 'string';

const numberFormats = new Map();
function memoizedNumberFormatter(locales, options) {
  // force a latin locale for number formatting
  const latnLocales = latinLocales(locales);
  const key = numberFormatCacheKey(latnLocales, options);

  if (numberFormats.has(key)) {
    return numberFormats.get(key);
  }

  const i = new Intl.NumberFormat(latnLocales, options);
  numberFormats.set(key, i);
  return i;
}

function latinLocales(locales) {
  return Array.isArray(locales) ? locales.map(locale => latinLocale(locale)) : latinLocale(locales);
}

function latinLocale(locale) {
  if (!locale) return locale; // Intl.Locale was added to iOS in v14. See https://caniuse.com/?search=Intl.Locale
  // We still support ios 12/13, so we need to check if this works and fallback to the default behaviour if not

  try {
    return new Intl.Locale(locale, {
      numberingSystem: LATIN
    }).toString();
  } catch {
    const numberingSystemRegex = new RegExp(`(?:-x|${UNICODE_NUMBERING_SYSTEM}).*`, 'g');
    const latinNumberingSystem = `${UNICODE_NUMBERING_SYSTEM}${LATIN}`;
    return locale.replace(numberingSystemRegex, '').concat(latinNumberingSystem);
  }
}

const PSEUDOTRANSLATE_OPTIONS = {
  startDelimiter: '{',
  endDelimiter: '}',
  prepend: '[!!',
  append: '!!]'
};
function numberFormatCacheKey(locales, options = {}) {
  const localeKey = Array.isArray(locales) ? locales.sort().join('-') : locales;
  return `${localeKey}-${JSON.stringify(options)}`;
}

function pluralRules(locale, options = {}) {
  return new Intl.PluralRules(locale, options);
}

const memoizedPluralRules = memoize(pluralRules, (locale, options = {}) => `${locale}${JSON.stringify(options)}`);
function getTranslationTree(id, translations, locale, replacements) {
  const normalizedTranslations = Array.isArray(translations) ? translations : [translations];
  let result;

  for (const translationDictionary of normalizedTranslations) {
    result = translationDictionary;

    for (const part of id.split(SEPARATOR)) {
      result = result[part];
      if (!result) break;
    }

    if (result) {
      if (replacements) {
        return isString(result) ? updateStringWithReplacements(result, replacements) : updateTreeWithReplacements(result, locale, replacements);
      }

      return result;
    }
  }

  throw new MissingTranslationError(id, locale);
}
function translate(id, options, translations, locale) {
  const {
    scope,
    replacements,
    pseudotranslate,
    interpolate
  } = options;
  const normalizedTranslations = Array.isArray(translations) ? translations : [translations];
  const normalizedId = normalizeIdentifier(id, scope);

  for (const translationDictionary of normalizedTranslations) {
    const result = translateWithDictionary(normalizedId, translationDictionary, locale, replacements, {
      pseudotranslate,
      interpolate
    });

    if (result !== MISSING_TRANSLATION) {
      return result;
    }
  }

  throw new MissingTranslationError(normalizedId, locale);
}

function translateWithDictionary(id, translations, locale, replacements, {
  pseudotranslate: pseudotranslate$1 = false,
  interpolate
} = {}) {
  let result = translations;

  for (const part of id.split(SEPARATOR)) {
    if (result == null || typeof result !== 'object') {
      return MISSING_TRANSLATION;
    }

    result = result[part];
  }

  const additionalReplacements = {};

  if (typeof result === 'object' && replacements != null && Object.prototype.hasOwnProperty.call(replacements, CARDINAL_PLURALIZATION_KEY_NAME)) {
    const count = replacements[CARDINAL_PLURALIZATION_KEY_NAME];

    if (typeof count === 'number') {
      // Explicit 0 and 1 rules take precedence over the pluralization rules
      // https://unicode-org.github.io/cldr/ldml/tr35-numbers.html#Explicit_0_1_rules
      if (count === 0 && result['0'] !== undefined) {
        result = result['0'];
      } else if (count === 1 && result['1'] !== undefined) {
        result = result['1'];
      } else {
        const group = memoizedPluralRules(locale).select(count);
        result = result[group] || result.other;
      }

      additionalReplacements[CARDINAL_PLURALIZATION_KEY_NAME] = memoizedNumberFormatter(locale).format(count);
    }
  } else if (typeof result === 'object' && replacements != null && Object.prototype.hasOwnProperty.call(replacements, ORDINAL_PLURALIZATION_KEY_NAME)) {
    const count = replacements[ORDINAL_PLURALIZATION_KEY_NAME];

    if (typeof count === 'number') {
      const group = memoizedPluralRules(locale, {
        type: 'ordinal'
      }).select(count);
      result = result.ordinal[group] || result.ordinal['other'];
      additionalReplacements[ORDINAL_PLURALIZATION_KEY_NAME] = memoizedNumberFormatter(locale).format(count);
    }
  }

  const processedString = isString(result) && pseudotranslate$1 ? pseudotranslate(result, { ...PSEUDOTRANSLATE_OPTIONS,
    toLocale: typeof pseudotranslate$1 === 'boolean' ? undefined : pseudotranslate$1
  }) : result;

  if (!isString(processedString)) {
    return MISSING_TRANSLATION;
  }

  return updateStringWithReplacements(processedString, { ...replacements,
    ...additionalReplacements
  }, {
    interpolate
  });
}

function updateStringWithReplacements(str, replacements = {}, {
  interpolate
} = {}) {
  const pieces = [];
  const replaceFinder = new RegExp(interpolate || DEFAULT_FORMAT, 'g');
  let matchIndex = 0;
  let lastOffset = 0; // Uses replace callback, but not its return value

  str.replace(replaceFinder, (match, replacementKey, offset) => {
    if (!replacementKey) {
      throw new Error('Invalid replacement key. The interpolatation format RegExp is possibly too permissive.');
    }

    if (!Object.prototype.hasOwnProperty.call(replacements, replacementKey)) {
      throw new MissingReplacementError(replacementKey, replacements);
    }

    matchIndex += 1; // Push the previous part if it exists

    const previousString = str.substring(lastOffset, offset);
    if (previousString) pieces.push(previousString);
    lastOffset = offset + match.length; // Push the new part with the replacement

    const replacementValue = replacements[replacementKey];

    if ( /*#__PURE__*/React.isValidElement(replacementValue)) {
      pieces.push( /*#__PURE__*/React.cloneElement(replacementValue, {
        key: matchIndex
      }));
    } else if (typeof replacementValue === 'object') {
      pieces.push(replacementValue);
    } else {
      pieces.push(String(replacementValue));
    } // to satisfy the typechecker


    return '';
  }); // Push the last part of the source string

  const lastPart = str.substring(lastOffset);
  if (lastPart) pieces.push(lastPart);
  return pieces.every(isString) ? pieces.join('') : pieces;
}

function normalizeIdentifier(id, scope) {
  if (scope == null) {
    return id;
  }

  return `${isString(scope) ? scope : scope.join(SEPARATOR)}${SEPARATOR}${id}`;
}

function updateTreeWithReplacements(translationTree, locale, replacements) {
  if (Object.prototype.hasOwnProperty.call(replacements, CARDINAL_PLURALIZATION_KEY_NAME)) {
    const count = replacements[CARDINAL_PLURALIZATION_KEY_NAME];

    if (typeof count === 'number') {
      const group = memoizedPluralRules(locale).select(count);

      if (isString(translationTree[group])) {
        return updateStringWithReplacements(translationTree[group], { ...replacements,
          CARDINAL_PLURALIZATION_KEY_NAME: memoizedNumberFormatter(locale).format(count)
        });
      }
    }
  } else if (Object.prototype.hasOwnProperty.call(replacements, ORDINAL_PLURALIZATION_KEY_NAME)) {
    const count = replacements[ORDINAL_PLURALIZATION_KEY_NAME];

    if (typeof count === 'number') {
      const group = memoizedPluralRules(locale, {
        type: 'ordinal'
      }).select(count);

      if (isString(translationTree[group])) {
        return updateStringWithReplacements(translationTree[group], { ...replacements,
          ORDINAL_PLURALIZATION_KEY_NAME: memoizedNumberFormatter(locale).format(count)
        });
      }
    }
  }

  return Object.keys(translationTree).reduce((acc, key) => ({ ...acc,
    [key]: isString(translationTree[key]) ? updateStringWithReplacements(translationTree[key], replacements) : updateTreeWithReplacements(translationTree[key], locale, replacements)
  }), {});
}

export { PSEUDOTRANSLATE_OPTIONS, getTranslationTree, memoizedNumberFormatter, memoizedPluralRules, numberFormatCacheKey, translate };
