import React, { useRef, useState, useEffect } from 'react';
import { ChevronDownMinor, CaretDownMinor, CancelSmallMinor } from '@shopify/polaris-icons';
import { useToggle } from '../../../../utilities/use-toggle.js';
import { useBreakpoints } from '../../../../utilities/breakpoints.js';
import { classNames } from '../../../../utilities/css.js';
import styles from './FilterPill.scss.js';
import { useI18n } from '../../../../utilities/i18n/hooks.js';
import { useFeatures } from '../../../../utilities/features/hooks.js';
import { Text } from '../../../Text/Text.js';
import { HorizontalStack } from '../../../HorizontalStack/HorizontalStack.js';
import { UnstyledButton } from '../../../UnstyledButton/UnstyledButton.js';
import { Icon } from '../../../Icon/Icon.js';
import { Button } from '../../../Button/Button.js';
import { Popover } from '../../../Popover/Popover.js';
import { VerticalStack } from '../../../VerticalStack/VerticalStack.js';

function FilterPill({
  filterKey,
  label,
  filter,
  disabled,
  hideClearButton,
  selected,
  initialActive,
  closeOnChildOverlayClick,
  onRemove,
  onClick
}) {
  const i18n = useI18n();
  const {
    mdDown
  } = useBreakpoints();
  const {
    polarisSummerEditions2023
  } = useFeatures();
  const elementRef = useRef(null);
  const {
    value: focused,
    setTrue: setFocusedTrue,
    setFalse: setFocusedFalse
  } = useToggle(false);
  const [popoverActive, setPopoverActive] = useState(initialActive);
  useEffect(() => {
    const node = elementRef.current;
    if (!node || !popoverActive) {
      return;
    }
    const parent = node.parentElement?.parentElement;
    if (!parent) {
      return;
    }
    parent.scroll?.({
      left: node.offsetLeft
    });
  }, [elementRef, popoverActive]);
  const togglePopoverActive = () => {
    if (filter) {
      setPopoverActive(popoverActive => !popoverActive);
    }
    if (onClick) {
      onClick(filterKey);
    }
  };
  const handleClear = () => {
    if (onRemove) onRemove(filterKey);
    setPopoverActive(false);
  };
  const buttonClasses = classNames(styles.FilterButton, selected && styles.ActiveFilterButton, popoverActive && styles.FocusFilterButton, focused && styles.focusedFilterButton, disabled && styles.disabledFilterButton);
  const clearButtonClassNames = classNames(styles.PlainButton, styles.clearButton);
  const toggleButtonClassNames = classNames(styles.PlainButton, styles.ToggleButton);
  const se23LabelVariant = mdDown && polarisSummerEditions2023 ? 'bodyLg' : 'bodySm';
  const labelVariant = mdDown ? 'bodyMd' : 'bodySm';
  const wrappedLabel = /*#__PURE__*/React.createElement("div", {
    className: styles.Label
  }, /*#__PURE__*/React.createElement(Text, {
    variant: polarisSummerEditions2023 ? se23LabelVariant : labelVariant,
    as: "span"
  }, label));
  const activator = /*#__PURE__*/React.createElement("div", {
    className: buttonClasses
  }, /*#__PURE__*/React.createElement(HorizontalStack, {
    gap: "0",
    wrap: false
  }, /*#__PURE__*/React.createElement(UnstyledButton, {
    onFocus: setFocusedTrue,
    onBlur: setFocusedFalse,
    onClick: togglePopoverActive,
    className: toggleButtonClassNames,
    type: "button",
    disabled: disabled
  }, /*#__PURE__*/React.createElement(HorizontalStack, {
    wrap: false,
    align: "center",
    blockAlign: "center",
    gap: "0"
  }, selected ? /*#__PURE__*/React.createElement(React.Fragment, null, wrappedLabel) : /*#__PURE__*/React.createElement(React.Fragment, null, wrappedLabel, /*#__PURE__*/React.createElement("div", {
    className: styles.IconWrapper
  }, /*#__PURE__*/React.createElement(Icon, {
    source: polarisSummerEditions2023 ? ChevronDownMinor : CaretDownMinor,
    color: "base"
  }))))), selected ? /*#__PURE__*/React.createElement(UnstyledButton, {
    onClick: handleClear,
    className: clearButtonClassNames,
    type: "button",
    "aria-label": i18n.translate('Polaris.FilterPill.clear'),
    disabled: disabled
  }, /*#__PURE__*/React.createElement("div", {
    className: styles.IconWrapper
  }, /*#__PURE__*/React.createElement(Icon, {
    source: CancelSmallMinor,
    color: "base"
  }))) : null));
  const clearButtonMarkup = !hideClearButton && /*#__PURE__*/React.createElement("div", {
    className: styles.ClearButtonWrapper
  }, /*#__PURE__*/React.createElement(Button, {
    onClick: handleClear,
    plain: true,
    disabled: !selected,
    textAlign: "left"
  }, i18n.translate('Polaris.FilterPill.clear')));
  if (disabled) {
    return null;
  }
  return /*#__PURE__*/React.createElement("div", {
    ref: elementRef
  }, /*#__PURE__*/React.createElement(Popover, {
    active: popoverActive && !disabled,
    activator: activator,
    key: filterKey,
    onClose: togglePopoverActive,
    preferredAlignment: "left",
    preventCloseOnChildOverlayClick: !closeOnChildOverlayClick
  }, /*#__PURE__*/React.createElement("div", {
    className: styles.PopoverWrapper
  }, /*#__PURE__*/React.createElement(Popover.Section, null, /*#__PURE__*/React.createElement(VerticalStack, {
    gap: "1"
  }, filter, clearButtonMarkup)))));
}

export { FilterPill };
