/**
 * Generates a list of times. If the provided `startDate` is the current day in the shop's TZ, the list will start at the current time.
 *
 * @param startDate - Date to start the list of times from.
 * @param disableTimesBefore - sets a lower bound on the times list if the date is the same as {@param startDate}
 * @returns A list of times that can be selected for the provided `startDate`.
 */
export declare function generateTimes(startDate: Date, disableTimesBefore?: Date): Date[];
/**
 * Verifies that a time string can be converted to a time value for a Date object.
 *
 * @param time - Time string to verify
 * @returns `true` if the time string can be converted to a time value for a Date object, `false` otherwise
 */
export declare function isValidTime(time?: string): boolean;
/**
 * Given a list of dates, returns a list of options that can be used by a {@link https://polaris.shopify.com/components/selection-and-input/select| @shopify/polaris select}
 *
 * @param dates - List of dates to convert to options
 * @param locale - Locale to use for date formatting
 * @param ianaTimezone - IANA timezone to use for date formatting
 * @returns List of options
 */
export declare function formatDateListAsOptionList(dates: Date[], locale: string, ianaTimezone: string): {
    value: string;
    label: string;
}[];
/**
 * Convenience function that wraps the {@link https://github.com/Shopify/quilt/tree/main/packages/dates#formatdate| @shopify/date formatDate} utility.
 *
 * @param date - Date to format
 * @param locale - Locale to use for formatting
 * @returns Time string formatted according to the provided locale
 */
export declare function getLocalizedTimeForDate(date: Date, locale: string): string;
/**
 * If the timeInput is a valid time, returns a new Date object with a calendar day of forDate and a time of timeInput.
 * Date is offset by the browser and shop (@param ianaTimezone) timezone.
 *
 * @param timeInput - A string representing a time.
 * @param forDate - A Date object representing the date to set the time on
 * @param ianaTimezone - The IANA timezone of the shop
 *
 * @returns A `Date` object if timeInput is valid, `null` otherwise
 */
export declare function getValidDateForTime(timeInput: string, forDate: Date, ianaTimezone: string): Date | null;
