import React, { useState, useCallback, useEffect } from 'react';
import { TextField, EventListener } from '@shopify/polaris';
import { usePrevious } from '@shopify/react-hooks';
import { useI18n } from '@shopify/react-i18n';

// The FormattedNumberField is _not_ a fully controlled component, and only calls the onChange

const FALLBACK_INPUT = '0';
const MINIMUM_FRACTION_DIGITS = 0;
function FormattedNumberField({
  value,
  format,
  unformat,
  noFormatting,
  autoComplete,
  placeholder,
  defaultToPlaceholder,
  onChange,
  onInput,
  onBlur,
  positiveOnly,
  minimumFractionDigits = MINIMUM_FRACTION_DIGITS,
  maximumFractionDigits,
  ...textFieldProps
}) {
  const [fieldValue, setFieldValue] = useState('');
  const [addEventListener, setAddEventListener] = useState(false);
  const [i18n] = useI18n();
  const previousValue = usePrevious(value);
  const handleChange = value => {
    onInput === null || onInput === void 0 ? void 0 : onInput(value);
    setFieldValue(value);
    if (!addEventListener) setAddEventListener(true);
  };
  const handleKeyPress = event => {
    if (event.key === 'Enter') {
      handleBlur();
    }
  };
  const handleBlur = () => {
    if (noFormatting) {
      return;
    }
    const unformattedNumber = unformat == null ? defaultUnformat(fieldValue) : unformat(fieldValue);
    const unformatted = positiveOnly ? unformattedNumber.replace('-', '') : unformattedNumber;
    const normalizedFieldValue = unformatted === '' && fieldValue !== '' ? FALLBACK_INPUT : unformatted;
    const newFieldValue = formatInput(normalizedFieldValue);
    setFieldValue(newFieldValue);
    setAddEventListener(false);
    if (onBlur) {
      onBlur();
    }
    if (maximumFractionDigits !== undefined) {
      const roundedNormalizedFieldValue = round(Number(normalizedFieldValue), maximumFractionDigits);
      onChange(String(roundedNormalizedFieldValue));
      return;
    }
    onChange(normalizedFieldValue);
  };
  const defaultFormat = useCallback(value => {
    if (maximumFractionDigits !== undefined) {
      return i18n.formatNumber(Number(value), {
        minimumFractionDigits,
        maximumFractionDigits: Math.max(maximumFractionDigits, minimumFractionDigits)
      });
    }
    return i18n.formatNumber(Number(value), {
      minimumFractionDigits
    });
  }, [i18n, maximumFractionDigits, minimumFractionDigits]);
  const defaultUnformat = value => {
    const unformattedNumber = i18n.unformatNumber(value);
    if (unformattedNumber === '' && defaultToPlaceholder && placeholder) {
      return i18n.unformatNumber(placeholder);
    } else {
      return unformattedNumber;
    }
  };
  const formatInput = useCallback(input => {
    if (noFormatting) {
      return input;
    }
    if (isNaN(Number(input)) || input === '') {
      return '';
    }
    return format == null ? defaultFormat(input) : format(input);
  }, [defaultFormat, format, noFormatting]);

  // Sync the "value" prop into state
  useEffect(() => {
    if (value !== previousValue && value !== fieldValue) {
      const formatted = value ? formatInput(value) : '';
      setFieldValue(formatted);
    }
  }, [value, fieldValue, formatInput, previousValue]);
  const onKeyPressListener = addEventListener ? /*#__PURE__*/React.createElement(EventListener, {
    event: "keypress",
    handler: handleKeyPress
  }) : null;
  return /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(TextField, Object.assign({}, textFieldProps, {
    autoComplete: autoComplete ? autoComplete : 'off',
    placeholder: placeholder == null ? formatInput(FALLBACK_INPUT) : placeholder,
    onChange: handleChange,
    onBlur: handleBlur,
    value: fieldValue
  })), onKeyPressListener);
}
function round(value, maximumFractionDigits) {
  const multiplier = 10 ** maximumFractionDigits;
  return Math.round(value * multiplier) / multiplier;
}

export { FormattedNumberField };
