import React from 'react';
import { LegacyCard, FormLayout, Checkbox } from '@shopify/polaris';
import { isSameDay } from '@shopify/dates';
import { useI18n } from '@shopify/react-i18n';
import { getDateTimeInShopTimeZone, getDateInUTC, getNewDateAtEndOfDay } from '../../utilities/dates.js';
import { DEFAULT_WEEK_START_DAY } from '../../constants.js';
import { DatePicker } from '../DatePicker/DatePicker.js';
import { TimePicker } from '../TimePicker/TimePicker.js';

function ActiveDatesCard({
  startDate,
  endDate,
  timezoneAbbreviation,
  weekStartsOn = DEFAULT_WEEK_START_DAY,
  disabled
}) {
  const [i18n] = useI18n();
  const nowInUTC = new Date();
  const ianaTimezone = i18n.defaultTimezone;
  const showEndDate = Boolean(endDate.value);

  // When start date or time changes, updates the end date to be later than start date (if applicable)
  const handleStartDateTimeChange = nextStart => {
    startDate.onChange(nextStart);
    if (endDate.value) {
      const nextEndDate = getValidEndDateTime(nextStart, endDate.value, ianaTimezone);
      if (nextEndDate !== endDate.value) {
        endDate.onChange(nextEndDate);
      }
    }
  };

  // When end date or time changes, verify that the new end date is later than the start time
  const handleEndDateTimeChange = requestedEndDate => {
    const nextEndDate = getValidEndDateTime(startDate.value, requestedEndDate, ianaTimezone);
    endDate.onChange(nextEndDate);
  };
  const handleShowEndDateChange = () => {
    if (showEndDate) {
      endDate.onChange(null);
    } else {
      const startDateInShopTZ = getDateTimeInShopTimeZone(startDate.value, ianaTimezone);
      const endDateAtEndOfDay = getDateInUTC(getNewDateAtEndOfDay(startDateInShopTZ), ianaTimezone);
      endDate.onChange(endDateAtEndOfDay.toISOString());
    }
  };
  const endDateIsStartDate = endDate.value && isSameDay(new Date(endDate.value), new Date(startDate.value));
  const disableEndDatesBefore = getEndDatePickerDisableDatesBefore(nowInUTC, new Date(startDate.value));
  return /*#__PURE__*/React.createElement(LegacyCard, {
    title: i18n.translate('DiscountAppComponents.ActiveDatesCard.title'),
    sectioned: true
  }, /*#__PURE__*/React.createElement(FormLayout, null, /*#__PURE__*/React.createElement(FormLayout.Group, null, /*#__PURE__*/React.createElement(DatePicker, {
    date: {
      ...startDate,
      onChange: handleStartDateTimeChange
    },
    weekStartsOn: weekStartsOn,
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.startDate'),
    disableDatesBefore: nowInUTC.toISOString()
  }), /*#__PURE__*/React.createElement(TimePicker, {
    time: {
      ...startDate,
      onChange: handleStartDateTimeChange
    },
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.startTime', {
      timezoneAbbreviation
    }),
    disableTimesBefore: nowInUTC.toISOString()
  })), /*#__PURE__*/React.createElement(FormLayout.Group, null, /*#__PURE__*/React.createElement(Checkbox, {
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.setEndDate'),
    checked: showEndDate,
    disabled: disabled,
    onChange: handleShowEndDateChange
  })), showEndDate && endDate.value && /*#__PURE__*/React.createElement(FormLayout.Group, null, /*#__PURE__*/React.createElement(DatePicker, {
    date: {
      ...endDate,
      onChange: handleEndDateTimeChange,
      error: endDateIsStartDate ? undefined : endDate.error
    },
    weekStartsOn: weekStartsOn,
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.endDate'),
    disableDatesBefore: disableEndDatesBefore.toISOString()
  }), /*#__PURE__*/React.createElement(TimePicker, {
    time: {
      ...endDate,
      onChange: handleEndDateTimeChange,
      error: endDateIsStartDate ? endDate.error : undefined
    },
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.endTime', {
      timezoneAbbreviation
    }),
    disableTimesBefore: disableEndDatesBefore.toISOString()
  }))));
}

/**
 * The end date picker is disabled before the current time or the startDate, whichever is later
 */
function getEndDatePickerDisableDatesBefore(now, startDate) {
  return now > startDate ? now : startDate;
}

/**
 * Given a start and end date in UTC, returns a new valid end date in UTC if start date is after end date
 */
function getValidEndDateTime(startDateTime, endDateTime, ianaTimezone) {
  const startDate = getDateTimeInShopTimeZone(startDateTime, ianaTimezone);
  const endDate = getDateTimeInShopTimeZone(endDateTime, ianaTimezone);
  return startDate >= endDate ? getDateInUTC(getNewDateAtEndOfDay(startDate), ianaTimezone).toISOString() : endDateTime;
}

export { ActiveDatesCard };
