'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var React = require('react');
var polaris = require('@shopify/polaris');
var reactI18n = require('@shopify/react-i18n');
var MinimumRequirementsCard$1 = require('./MinimumRequirementsCard.scss.js');
var numbers = require('../../utilities/numbers.js');
var constants = require('../../constants.js');
var CurrencyField = require('../CurrencyField/CurrencyField.js');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var React__default = /*#__PURE__*/_interopDefaultLegacy(React);

const SUBTOTAL_FIELD_ID = 'discountSubtotal';
const QUANTITY_FIELD_ID = 'discountQuantity';
const AUTOMATIC_REQUIREMENT_TYPES = [constants.RequirementType.Subtotal, constants.RequirementType.Quantity];

// see https://github.com/Shopify/web/pull/12361
const SUBTOTAL_MAX_LENGTH = 15;

// see https://github.com/Shopify/web/pull/11506
const QUANTITY_MAX_LENGTH = 10;
function MinimumRequirementsCard({
  appliesTo,
  currencyCode,
  requirementType,
  subtotal,
  quantity,
  discountMethod,
  isRecurring = false
}) {
  const [i18n] = reactI18n.useI18n();

  // RequirementType.None is not valid for an automatic discount, so if the discount method changes we adjust the requirementType if the new state is invalid.
  React.useEffect(() => {
    if (discountMethod === constants.DiscountMethod.Automatic && requirementType.value === constants.RequirementType.None) {
      requirementType.onChange(constants.RequirementType.Subtotal);
    }
  }, [discountMethod, requirementType]);
  const currencySymbol = i18n.getCurrencySymbol(currencyCode).symbol;
  const minimumPurchaseLabel = i18n.translate('DiscountAppComponents.MinimumRequirementsCard.minimumSubtotal', {
    currencySymbol: currencySymbol.trim()
  });
  const minimumQuantityLabel = i18n.translate('DiscountAppComponents.MinimumRequirementsCard.minimumQuantity');
  const fieldHelpTextMarkup = /*#__PURE__*/React__default["default"].createElement(polaris.Text, {
    as: "span",
    color: "subdued"
  }, getFieldHelpText(isRecurring, appliesTo, i18n));
  const allMinimumRequirementChoices = [{
    label: i18n.translate('DiscountAppComponents.MinimumRequirementsCard.none'),
    value: constants.RequirementType.None
  }, {
    label: minimumPurchaseLabel,
    value: constants.RequirementType.Subtotal,
    renderChildren: isSelected => {
      return isSelected && /*#__PURE__*/React__default["default"].createElement(polaris.LegacyStack, {
        vertical: true,
        spacing: "extraTight"
      }, /*#__PURE__*/React__default["default"].createElement("div", {
        className: MinimumRequirementsCard$1["default"].TextField
      }, /*#__PURE__*/React__default["default"].createElement(CurrencyField["default"], {
        id: SUBTOTAL_FIELD_ID,
        label: minimumPurchaseLabel,
        labelHidden: true,
        value: subtotal.value,
        maxLength: SUBTOTAL_MAX_LENGTH,
        error: Boolean(subtotal.error),
        currencyCode: currencyCode,
        onChange: subtotal.onChange,
        onBlur: subtotal.onBlur,
        positiveOnly: true
      })), fieldHelpTextMarkup, subtotal.error && /*#__PURE__*/React__default["default"].createElement(polaris.InlineError, {
        fieldID: SUBTOTAL_FIELD_ID,
        message: subtotal.error
      }));
    }
  }, {
    label: minimumQuantityLabel,
    value: constants.RequirementType.Quantity,
    renderChildren: isSelected => {
      return isSelected && /*#__PURE__*/React__default["default"].createElement(polaris.LegacyStack, {
        vertical: true,
        spacing: "extraTight"
      }, /*#__PURE__*/React__default["default"].createElement("div", {
        className: MinimumRequirementsCard$1["default"].TextField
      }, /*#__PURE__*/React__default["default"].createElement(polaris.TextField, {
        id: QUANTITY_FIELD_ID,
        label: minimumQuantityLabel,
        labelHidden: true,
        autoComplete: "off",
        error: Boolean(quantity.error),
        value: quantity.value,
        maxLength: QUANTITY_MAX_LENGTH,
        onChange: nextValue => quantity.onChange(numbers.forcePositiveInteger(nextValue)),
        onBlur: quantity.onBlur
      })), fieldHelpTextMarkup, quantity.error && /*#__PURE__*/React__default["default"].createElement(polaris.InlineError, {
        fieldID: QUANTITY_FIELD_ID,
        message: quantity.error
      }));
    }
  }];
  const minimumRequirementChoicesToRender = discountMethod === constants.DiscountMethod.Automatic ? allMinimumRequirementChoices.filter(({
    value
  }) => AUTOMATIC_REQUIREMENT_TYPES.includes(value)) : allMinimumRequirementChoices;
  return /*#__PURE__*/React__default["default"].createElement(polaris.LegacyCard, {
    title: i18n.translate('DiscountAppComponents.MinimumRequirementsCard.title'),
    sectioned: true
  }, /*#__PURE__*/React__default["default"].createElement(polaris.ChoiceList, {
    title: i18n.translate('DiscountAppComponents.MinimumRequirementsCard.title'),
    titleHidden: true,
    choices: minimumRequirementChoicesToRender,
    selected: [requirementType.value],
    onChange: nextValue => requirementType.onChange(nextValue[0])
  }));
}
function getFieldHelpText(isRecurring, appliesTo, i18n) {
  const scope = isRecurring ? 'DiscountAppComponents.MinimumRequirementsCard.subscriptions' : 'DiscountAppComponents.MinimumRequirementsCard.oneTime';
  switch (appliesTo) {
    case constants.AppliesTo.Order:
      return i18n.translate('appliesToAllProducts', {
        scope
      });
    case constants.AppliesTo.Products:
      return i18n.translate('appliesToProducts', {
        scope
      });
    case constants.AppliesTo.Collections:
      return i18n.translate('appliesToCollections', {
        scope
      });
  }
}

exports.MinimumRequirementsCard = MinimumRequirementsCard;
