'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var React = require('react');
var polaris = require('@shopify/polaris');
var polarisIcons = require('@shopify/polaris-icons');
var d3TimeFormat = require('d3-time-format');
var reactI18n = require('@shopify/react-i18n');
var constants = require('../../constants.js');
var dates = require('../../utilities/dates.js');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var React__default = /*#__PURE__*/_interopDefaultLegacy(React);

const DATE_BLOCKLIST_REGEX = /[^\d-]/g;
const VALID_DATE_REGEX = /^\d{4}-\d{1,2}-\d{1,2}$/;
function DatePicker({
  date,
  label,
  labelHidden = false,
  disabled = false,
  disableDatesBefore,
  weekStartsOn = constants.DEFAULT_WEEK_START_DAY
}) {
  const [i18n] = reactI18n.useI18n();
  const ianaTimezone = i18n.defaultTimezone;
  const selectedDate = dates.getDateTimeInShopTimeZone(date.value, ianaTimezone);
  const localeFormattedDate = getFormattedDate(selectedDate);
  const [datePickerView, setDatePickerView] = React.useState({
    month: selectedDate.getMonth(),
    year: selectedDate.getFullYear()
  });
  const [userInput, setUserInput] = React.useState(localeFormattedDate);
  const [error, setError] = React.useState();
  const [isActive, setIsActive] = React.useState(false);
  const disableDatesBeforeInShopTZAtStartOfDay = React.useMemo(() => {
    return disableDatesBefore ? dates.getNewDateAtStartOfDay(dates.getDateTimeInShopTimeZone(disableDatesBefore, ianaTimezone)) : undefined;
  }, [disableDatesBefore, ianaTimezone]);
  React.useEffect(() => {
    const selectedDate = dates.getDateTimeInShopTimeZone(date.value, ianaTimezone);
    setDatePickerView({
      month: selectedDate.getMonth(),
      year: selectedDate.getFullYear()
    });
    setUserInput(getFormattedDate(selectedDate));
  }, [date.value, ianaTimezone]);
  const handleMonthChange = (month, year) => {
    setDatePickerView({
      month,
      year
    });
  };
  const handleTextFieldChange = inputValue => setUserInput(inputValue.replace(DATE_BLOCKLIST_REGEX, ''));
  const handleTextFieldBlur = () => {
    var _date$onBlur;
    if (userInput === localeFormattedDate) {
      return;
    }
    setError(null);
    if (userInput === '') {
      setUserInput(localeFormattedDate);
    } else if (isValidDateString(userInput)) {
      const newDate = d3TimeFormat.timeParse('%Y-%m-%d')(userInput);
      const hasDisabledDatesAndInputIsAfterDisabledDate = disableDatesBeforeInShopTZAtStartOfDay && newDate >= disableDatesBeforeInShopTZAtStartOfDay;
      if (!disableDatesBeforeInShopTZAtStartOfDay || hasDisabledDatesAndInputIsAfterDisabledDate) {
        setDatePickerView({
          month: newDate.getMonth(),
          year: newDate.getFullYear()
        });
        handleDateChange(dates.getDateTimeInShopTimeZone(date.value, ianaTimezone), newDate, date.onChange, ianaTimezone);
      } else {
        setUserInput(localeFormattedDate);
      }
    } else {
      setError(i18n.translate('DiscountAppComponents.DatePicker.dateFormatError'));
    }
    (_date$onBlur = date.onBlur) === null || _date$onBlur === void 0 ? void 0 : _date$onBlur.call(date);
  };
  const handleDatePickerChange = value => {
    const formattedDate = getFormattedDate(value.start);
    setError(null);
    setUserInput(formattedDate);
    setIsActive(false);
    setDatePickerView({
      month: value.start.getMonth(),
      year: value.start.getFullYear()
    });
    handleDateChange(dates.getDateTimeInShopTimeZone(date.value, ianaTimezone), value.start, date.onChange, ianaTimezone);
  };
  return /*#__PURE__*/React__default["default"].createElement(polaris.Popover, {
    activator: /*#__PURE__*/React__default["default"].createElement(polaris.TextField, {
      value: userInput,
      label: label,
      labelHidden: labelHidden,
      prefix: /*#__PURE__*/React__default["default"].createElement(polaris.Icon, {
        source: polarisIcons.CalendarMajor,
        color: "subdued"
      }),
      placeholder: i18n.translate('DiscountAppComponents.DatePicker.datePlaceholder'),
      error: date.error || error || false,
      autoComplete: "off",
      onFocus: () => setIsActive(true),
      onChange: handleTextFieldChange,
      onBlur: handleTextFieldBlur,
      disabled: disabled
    }),
    active: isActive,
    onClose: () => setIsActive(false),
    autofocusTarget: "none",
    sectioned: true
  }, /*#__PURE__*/React__default["default"].createElement(polaris.DatePicker, {
    month: datePickerView.month,
    year: datePickerView.year,
    disableDatesBefore: disableDatesBeforeInShopTZAtStartOfDay,
    weekStartsOn: weekStartsOn,
    onMonthChange: handleMonthChange,
    selected: selectedDate,
    onChange: handleDatePickerChange
  }));
}
function isValidDateString(date) {
  if (!date) {
    return false;
  }
  return VALID_DATE_REGEX.test(date) && !isNaN(new Date(date).getTime());
}
const getFormattedDate = selectedDate => d3TimeFormat.timeFormat('%Y-%m-%d')(selectedDate);
const handleDateChange = (oldDate, newDate, onChange, ianaTimezone) => {
  // Re-apply the current time before firing the datetime field onChange.
  newDate.setHours(oldDate.getHours(), oldDate.getMinutes(), oldDate.getSeconds(), oldDate.getMilliseconds());
  if (oldDate.getTime() !== newDate.getTime()) {
    // Persist date in UTC
    onChange(dates.getDateInUTC(newDate, ianaTimezone).toISOString());
  }
};

exports.DatePicker = DatePicker;
