'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var React = require('react');
var polaris = require('@shopify/polaris');
var dates = require('@shopify/dates');
var reactI18n = require('@shopify/react-i18n');
var dates$1 = require('../../utilities/dates.js');
var constants = require('../../constants.js');
var DatePicker = require('../DatePicker/DatePicker.js');
var TimePicker = require('../TimePicker/TimePicker.js');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var React__default = /*#__PURE__*/_interopDefaultLegacy(React);

function ActiveDatesCard({
  startDate,
  endDate,
  timezoneAbbreviation,
  weekStartsOn = constants.DEFAULT_WEEK_START_DAY,
  disabled
}) {
  const [i18n] = reactI18n.useI18n();
  const nowInUTC = new Date();
  const ianaTimezone = i18n.defaultTimezone;
  const showEndDate = Boolean(endDate.value);

  // When start date or time changes, updates the end date to be later than start date (if applicable)
  const handleStartDateTimeChange = nextStart => {
    startDate.onChange(nextStart);
    if (endDate.value) {
      const nextEndDate = getValidEndDateTime(nextStart, endDate.value, ianaTimezone);
      if (nextEndDate !== endDate.value) {
        endDate.onChange(nextEndDate);
      }
    }
  };

  // When end date or time changes, verify that the new end date is later than the start time
  const handleEndDateTimeChange = requestedEndDate => {
    const nextEndDate = getValidEndDateTime(startDate.value, requestedEndDate, ianaTimezone);
    endDate.onChange(nextEndDate);
  };
  const handleShowEndDateChange = () => {
    if (showEndDate) {
      endDate.onChange(null);
    } else {
      const startDateInShopTZ = dates$1.getDateTimeInShopTimeZone(startDate.value, ianaTimezone);
      const endDateAtEndOfDay = dates$1.getDateInUTC(dates$1.getNewDateAtEndOfDay(startDateInShopTZ), ianaTimezone);
      endDate.onChange(endDateAtEndOfDay.toISOString());
    }
  };
  const endDateIsStartDate = endDate.value && dates.isSameDay(new Date(endDate.value), new Date(startDate.value));
  const disableEndDatesBefore = getEndDatePickerDisableDatesBefore(nowInUTC, new Date(startDate.value));
  return /*#__PURE__*/React__default["default"].createElement(polaris.LegacyCard, {
    title: i18n.translate('DiscountAppComponents.ActiveDatesCard.title'),
    sectioned: true
  }, /*#__PURE__*/React__default["default"].createElement(polaris.FormLayout, null, /*#__PURE__*/React__default["default"].createElement(polaris.FormLayout.Group, null, /*#__PURE__*/React__default["default"].createElement(DatePicker.DatePicker, {
    date: {
      ...startDate,
      onChange: handleStartDateTimeChange
    },
    weekStartsOn: weekStartsOn,
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.startDate'),
    disableDatesBefore: nowInUTC.toISOString()
  }), /*#__PURE__*/React__default["default"].createElement(TimePicker.TimePicker, {
    time: {
      ...startDate,
      onChange: handleStartDateTimeChange
    },
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.startTime', {
      timezoneAbbreviation
    }),
    disableTimesBefore: nowInUTC.toISOString()
  })), /*#__PURE__*/React__default["default"].createElement(polaris.FormLayout.Group, null, /*#__PURE__*/React__default["default"].createElement(polaris.Checkbox, {
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.setEndDate'),
    checked: showEndDate,
    disabled: disabled,
    onChange: handleShowEndDateChange
  })), showEndDate && endDate.value && /*#__PURE__*/React__default["default"].createElement(polaris.FormLayout.Group, null, /*#__PURE__*/React__default["default"].createElement(DatePicker.DatePicker, {
    date: {
      ...endDate,
      onChange: handleEndDateTimeChange,
      error: endDateIsStartDate ? undefined : endDate.error
    },
    weekStartsOn: weekStartsOn,
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.endDate'),
    disableDatesBefore: disableEndDatesBefore.toISOString()
  }), /*#__PURE__*/React__default["default"].createElement(TimePicker.TimePicker, {
    time: {
      ...endDate,
      onChange: handleEndDateTimeChange,
      error: endDateIsStartDate ? endDate.error : undefined
    },
    disabled: disabled,
    label: i18n.translate('DiscountAppComponents.ActiveDatesCard.endTime', {
      timezoneAbbreviation
    }),
    disableTimesBefore: disableEndDatesBefore.toISOString()
  }))));
}

/**
 * The end date picker is disabled before the current time or the startDate, whichever is later
 */
function getEndDatePickerDisableDatesBefore(now, startDate) {
  return now > startDate ? now : startDate;
}

/**
 * Given a start and end date in UTC, returns a new valid end date in UTC if start date is after end date
 */
function getValidEndDateTime(startDateTime, endDateTime, ianaTimezone) {
  const startDate = dates$1.getDateTimeInShopTimeZone(startDateTime, ianaTimezone);
  const endDate = dates$1.getDateTimeInShopTimeZone(endDateTime, ianaTimezone);
  return startDate >= endDate ? dates$1.getDateInUTC(dates$1.getNewDateAtEndOfDay(startDate), ianaTimezone).toISOString() : endDateTime;
}

exports.ActiveDatesCard = ActiveDatesCard;
