import { memoizedGetDateTimeFormat } from './utilities/formatDate.mjs';

const TRANSFORM_MAP = {
  numericFromTwoDigit: time => time.startsWith('0') && time.length === 2 ? time.charAt(1) : time,
  twoDigitFromNumeric: time => time.length === 1 ? `0${time}` : time,
  lowercase: time => time.toLowerCase()
};
const TOKEN_MAP = {
  YYYY: {
    type: 'year',
    options: {
      year: 'numeric'
    }
  },
  YY: {
    type: 'year',
    options: {
      year: '2-digit'
    }
  },
  MMMM: {
    type: 'month',
    options: {
      month: 'long'
    }
  },
  MMM: {
    type: 'month',
    options: {
      month: 'short'
    }
  },
  MM: {
    type: 'month',
    options: {
      month: '2-digit'
    }
  },
  // eslint-disable-next-line id-length
  M: {
    type: 'month',
    options: {
      month: 'numeric'
    }
  },
  DD: {
    type: 'day',
    options: {
      day: '2-digit'
    }
  },
  // eslint-disable-next-line id-length
  D: {
    type: 'day',
    options: {
      day: 'numeric'
    }
  },
  HH: {
    type: 'hour',
    options: {
      hour: '2-digit',
      hour12: false
    }
  },
  // eslint-disable-next-line id-length
  H: {
    type: 'hour',
    options: {
      hour: 'numeric',
      hour12: false
    },
    // extraTransform is needed when Node and Browsers don't respect the 2-digit
    // and numeric options, which happens occasionally for time units
    extraTransform: TRANSFORM_MAP.numericFromTwoDigit
  },
  hh: {
    type: 'hour',
    options: {
      hour: '2-digit',
      hour12: true
    }
  },
  // eslint-disable-next-line id-length
  h: {
    type: 'hour',
    options: {
      hour: 'numeric',
      hour12: true
    }
  },
  mm: {
    type: 'minute',
    options: {
      minute: '2-digit'
    },
    // see 'H' token parsing for more details about extraTransform
    extraTransform: TRANSFORM_MAP.twoDigitFromNumeric
  },
  // eslint-disable-next-line id-length
  m: {
    type: 'minute',
    options: {
      minute: 'numeric'
    }
  },
  ss: {
    type: 'second',
    options: {
      second: '2-digit'
    },
    extraTransform: TRANSFORM_MAP.twoDigitFromNumeric
  },
  // eslint-disable-next-line id-length
  s: {
    type: 'second',
    options: {
      second: 'numeric'
    }
  },
  // eslint-disable-next-line id-length
  A: {
    type: 'dayPeriod',
    options: {
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      hour12: true
    }
  },
  // eslint-disable-next-line id-length
  a: {
    type: 'dayPeriod',
    options: {
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      hour12: true
    },
    extraTransform: TRANSFORM_MAP.lowercase
  },
  dddd: {
    type: 'weekday',
    options: {
      weekday: 'long'
    }
  },
  ddd: {
    type: 'weekday',
    options: {
      weekday: 'short'
    }
  }
};
const ALL_TOKENS_REGEX = new RegExp(`(${Object.keys(TOKEN_MAP).join('|')})`, 'g');
function format(date, fmtStr, tz, localeStr) {
  const {
    locale,
    timeZone
  } = memoizedGetDateTimeFormat(localeStr, {
    timeZone: tz
  }).resolvedOptions();
  return fmtStr.replace(ALL_TOKENS_REGEX, token => {
    var _Intl$DateTimeFormat$;

    const {
      type,
      options,
      extraTransform
    } = TOKEN_MAP[token];
    const timeUnit = (_Intl$DateTimeFormat$ = Intl.DateTimeFormat(locale, {
      timeZone,
      ...options
    }).formatToParts(date).find(part => part.type === type)) === null || _Intl$DateTimeFormat$ === void 0 ? void 0 : _Intl$DateTimeFormat$.value;

    if (timeUnit !== undefined && extraTransform) {
      return extraTransform(timeUnit);
    } else {
      return timeUnit !== null && timeUnit !== void 0 ? timeUnit : '';
    }
  });
}

export { format };
