import { version as rubyVersion } from './ruby.js';
import { alwaysLogAnalytics, analyticsDisabled, isShopify } from './context/local.js';
import * as metadata from './metadata.js';
import { publishMonorailEvent, MONORAIL_COMMAND_TOPIC } from './monorail.js';
import { fanoutHooks } from './plugins.js';
import { outputContent, outputDebug, outputToken } from '../../public/node/output.js';
import { getEnvironmentData, getSensitiveEnvironmentData } from '../../private/node/analytics.js';
import { CLI_KIT_VERSION } from '../common/version.js';
/**
 * Report an analytics event, sending it off to Monorail -- Shopify's internal analytics service.
 *
 * The payload for an event includes both generic data, and data gathered from installed plug-ins.
 *
 */
export async function reportAnalyticsEvent(options) {
    try {
        const payload = await buildPayload(options);
        if (payload === undefined) {
            // Nothing to log
            return;
        }
        if (!alwaysLogAnalytics() && analyticsDisabled()) {
            outputDebug(outputContent `Skipping command analytics, payload: ${outputToken.json(payload)}`);
            return;
        }
        const response = await publishMonorailEvent(MONORAIL_COMMAND_TOPIC, payload.public, payload.sensitive);
        if (response.type === 'error') {
            outputDebug(response.message);
        }
        // eslint-disable-next-line no-catch-all/no-catch-all
    }
    catch (error) {
        let message = 'Failed to report usage analytics';
        if (error instanceof Error) {
            message = message.concat(`: ${error.message}`);
        }
        outputDebug(message);
    }
}
async function buildPayload({ config, errorMessage }) {
    const { commandStartOptions, environmentFlags, ...sensitiveMetadata } = metadata.getAllSensitiveMetadata();
    if (commandStartOptions === undefined) {
        outputDebug('Unable to log analytics event - no information on executed command');
        return;
    }
    const { startCommand, startArgs, startTime } = commandStartOptions;
    const currentTime = new Date().getTime();
    const { '@shopify/app': appPublic, ...otherPluginsPublic } = await fanoutHooks(config, 'public_command_metadata', {});
    const { '@shopify/app': appSensitive, ...otherPluginsSensitive } = await fanoutHooks(config, 'sensitive_command_metadata', {});
    const environmentData = await getEnvironmentData(config);
    const sensitiveEnvironmentData = await getSensitiveEnvironmentData(config);
    let payload = {
        public: {
            command: startCommand,
            time_start: startTime,
            time_end: currentTime,
            total_time: currentTime - startTime,
            success: errorMessage === undefined,
            cli_version: CLI_KIT_VERSION,
            ruby_version: (await rubyVersion()) || '',
            node_version: process.version.replace('v', ''),
            is_employee: await isShopify(),
            ...environmentData,
            ...appPublic,
            ...metadata.getAllPublicMetadata(),
        },
        sensitive: {
            args: startArgs.join(' '),
            cmd_all_environment_flags: environmentFlags,
            error_message: errorMessage,
            ...appSensitive,
            ...sensitiveEnvironmentData,
            metadata: JSON.stringify({
                ...sensitiveMetadata,
                extraPublic: {
                    ...otherPluginsPublic,
                },
                extraSensitive: { ...otherPluginsSensitive },
            }),
        },
    };
    // strip undefined fields -- they make up the majority of payloads due to wide metadata structure.
    payload = JSON.parse(JSON.stringify(payload));
    return sanitizePayload(payload);
}
function sanitizePayload(payload) {
    const payloadString = JSON.stringify(payload);
    // Remove Theme Access passwords from the payload
    const sanitizedPayloadString = payloadString.replace(/shptka_\w*/g, '*****');
    return JSON.parse(sanitizedPayloadString);
}
//# sourceMappingURL=analytics.js.map