/* eslint-disable require-atomic-updates */
import { uploadFunctionExtensions, uploadThemeExtensions, uploadExtensionsBundle, } from './deploy/upload.js';
import { ensureDeployContext } from './context.js';
import { bundleAndBuildExtensions } from './deploy/bundle.js';
import { fetchAppExtensionRegistrations } from './dev/fetch.js';
import { updateAppIdentifiers } from '../models/app/identifiers.js';
import { renderInfo, renderSuccess, renderTasks } from '@shopify/cli-kit/node/ui';
import { inTemporaryDirectory, mkdir } from '@shopify/cli-kit/node/fs';
import { joinPath, dirname } from '@shopify/cli-kit/node/path';
import { outputNewline, outputInfo, formatPackageManagerCommand } from '@shopify/cli-kit/node/output';
import { useThemebundling } from '@shopify/cli-kit/node/context/local';
import { getArrayRejectingUndefined } from '@shopify/cli-kit/common/array';
export async function deploy(options) {
    // eslint-disable-next-line prefer-const
    let { app, identifiers, partnersApp, token, deploymentMode } = await ensureDeployContext(options);
    const apiKey = identifiers.app;
    const unifiedDeployment = deploymentMode !== 'legacy';
    if (!options.app.hasExtensions() && !unifiedDeployment) {
        renderInfo({ headline: 'No extensions to deploy to Shopify Partners yet.' });
        return;
    }
    outputNewline();
    switch (deploymentMode) {
        case 'legacy':
            outputInfo(`Deploying your work to Shopify Partners.`);
            break;
        case 'unified':
            outputInfo(`Releasing a new app version as part of ${partnersApp.title}`);
            break;
        case 'unified-skip-release':
            outputInfo(`Creating a new app version as part of ${partnersApp.title}`);
            break;
    }
    outputNewline();
    let registrations;
    let uploadExtensionsBundleResult;
    await inTemporaryDirectory(async (tmpDir) => {
        try {
            const bundle = app.allExtensions.some((ext) => ext.features.includes('bundling'));
            let bundlePath;
            if (bundle) {
                bundlePath = joinPath(tmpDir, `bundle.zip`);
                await mkdir(dirname(bundlePath));
            }
            await bundleAndBuildExtensions({ app, bundlePath, identifiers });
            const uploadTaskTitle = (() => {
                switch (deploymentMode) {
                    case 'legacy':
                        return 'Pushing your code to Shopify';
                    case 'unified':
                        return 'Releasing an app version';
                    case 'unified-skip-release':
                        return 'Creating an app version';
                }
            })();
            const tasks = [
                {
                    title: 'Running validation',
                    task: async () => {
                        await Promise.all([app.allExtensions.map((ext) => ext.preDeployValidation())]);
                    },
                },
                {
                    title: uploadTaskTitle,
                    task: async () => {
                        const appModules = await Promise.all(options.app.allExtensions.flatMap((ext) => ext.bundleConfig({ identifiers, token, apiKey, unifiedDeployment })));
                        if (bundle || unifiedDeployment) {
                            uploadExtensionsBundleResult = await uploadExtensionsBundle({
                                apiKey,
                                bundlePath,
                                appModules: getArrayRejectingUndefined(appModules),
                                deploymentMode,
                                token,
                                extensionIds: identifiers.extensionIds,
                                message: options.message,
                                version: options.version,
                                commitReference: options.commitReference,
                            });
                        }
                        if (!useThemebundling()) {
                            const themeExtensions = options.app.allExtensions.filter((ext) => ext.isThemeExtension);
                            await uploadThemeExtensions(themeExtensions, { apiKey, identifiers, token });
                        }
                        if (!unifiedDeployment) {
                            const functions = options.app.allExtensions.filter((ext) => ext.isFunctionExtension);
                            identifiers = await uploadFunctionExtensions(functions, {
                                identifiers,
                                token,
                            });
                        }
                        app = await updateAppIdentifiers({ app, identifiers, command: 'deploy' });
                        registrations = await fetchAppExtensionRegistrations({ token, apiKey: identifiers.app });
                    },
                },
            ];
            await renderTasks(tasks);
            await outputCompletionMessage({
                app,
                partnersApp,
                partnersOrganizationId: partnersApp.organizationId,
                identifiers,
                registrations,
                deploymentMode,
                uploadExtensionsBundleResult,
            });
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
        }
        catch (error) {
            /**
             * If deployment fails when uploading we want the identifiers to be persisted
             * for the next run.
             */
            await updateAppIdentifiers({ app, identifiers, command: 'deploy' });
            throw error;
        }
    });
}
async function outputCompletionMessage({ app, partnersApp, partnersOrganizationId, identifiers, registrations, deploymentMode, uploadExtensionsBundleResult, }) {
    if (deploymentMode !== 'legacy') {
        return outputUnifiedCompletionMessage(deploymentMode, uploadExtensionsBundleResult, app);
    }
    let headline;
    const validationErrors = uploadExtensionsBundleResult?.validationErrors ?? [];
    if (validationErrors.length > 0) {
        headline = 'Deployed to Shopify, but fixes are needed.';
    }
    else {
        headline = 'Deployed to Shopify!';
    }
    const outputDeployedButNotLiveMessage = (extension) => {
        const result = [`${extension.localIdentifier} is deployed to Shopify but not yet live`];
        const uuid = identifiers.extensions[extension.localIdentifier];
        const validationError = validationErrors.find((error) => error.uuid === uuid);
        if (validationError) {
            result.push('\n- Validation errors found in your extension toml file');
            validationError.errors.forEach((err) => {
                result.push(`\n  └ ${err.message}`);
            });
        }
        return result;
    };
    const outputDeployedAndLiveMessage = (extension) => {
        return `${extension.localIdentifier} is live`;
    };
    const outputNextStep = async (extension) => {
        const extensionId = registrations.app.extensionRegistrations.find((registration) => {
            return registration.uuid === identifiers.extensions[extension.localIdentifier];
        })?.id ?? '';
        return [
            'Publish',
            {
                link: {
                    url: await extension.publishURL({ orgId: partnersOrganizationId, appId: partnersApp.id, extensionId }),
                    label: extension.localIdentifier,
                },
            },
        ];
    };
    const nonFunctionExtensions = app.allExtensions.filter((ext) => !ext.isFunctionExtension);
    const functionExtensions = app.allExtensions.filter((ext) => ext.isFunctionExtension);
    const customSections = [
        {
            title: 'Summary',
            body: {
                list: {
                    items: [
                        ...nonFunctionExtensions.map(outputDeployedButNotLiveMessage),
                        ...functionExtensions.map(outputDeployedAndLiveMessage),
                    ],
                },
            },
        },
    ];
    if (nonFunctionExtensions.length > 0) {
        customSections.push({
            title: 'Next steps',
            body: {
                list: {
                    items: await Promise.all(nonFunctionExtensions.map(outputNextStep)),
                },
            },
        });
    }
    renderSuccess({
        headline,
        customSections,
    });
}
async function outputUnifiedCompletionMessage(deploymentMode, uploadExtensionsBundleResult, app) {
    const linkAndMessage = [
        { link: { label: uploadExtensionsBundleResult.versionTag, url: uploadExtensionsBundleResult.location } },
        uploadExtensionsBundleResult.message ? `\n${uploadExtensionsBundleResult.message}` : '',
    ];
    if (deploymentMode === 'unified') {
        return uploadExtensionsBundleResult.deployError
            ? renderInfo({
                headline: 'New version created, but not released.',
                body: [...linkAndMessage, `\n\n${uploadExtensionsBundleResult.deployError}`],
            })
            : renderSuccess({
                headline: 'New version released to users.',
                body: linkAndMessage,
            });
    }
    return renderSuccess({
        headline: 'New version created.',
        body: linkAndMessage,
        nextSteps: [
            [
                'Run',
                {
                    command: formatPackageManagerCommand(app.packageManager, 'shopify app release', `--version=${uploadExtensionsBundleResult.versionTag}`),
                },
                'to release this version to users.',
            ],
        ],
    });
}
//# sourceMappingURL=deploy.js.map