import { BaseSchema } from './schemas.js';
import { blocks, defaultExtensionFlavors } from '../../constants.js';
import { capitalize } from '@shopify/cli-kit/common/string';
/**
 * Create a new ui extension spec.
 *
 * Everything but "identifer" is optional.
 * ```ts
 * identifier: string // unique identifier for the extension type
 * externalIdentifier: string // identifier used externally (default: same as "identifier")
 * partnersWebIdentifier: string // identifier used in the partners web UI (default: same as "identifier")
 * surface?: string // surface where the extension is going to be rendered (default: 'unknown')
 * supportedFlavors: {name: string; value: string}[] // list of supported flavors (default: 'javascript', 'typescript', 'typescript-react', 'javascript-react')
 * helpURL?: string // url to the help page for the extension, shown after generating the extension
 * dependency?: {name: string; version: string} // dependency to be added to the extension's package.json
 * graphQLType?: string // GraphQL type of the extension (default: same as "identifier")
 * schema?: ZodSchemaType<TConfiguration> // schema used to validate the extension's configuration (default: BaseUIExtensionSchema)
 * getBundleExtensionStdinContent?: (configuration: TConfiguration) => string // function to generate the content of the stdin file used to bundle the extension
 * validate?: (configuration: TConfiguration, directory: string) => Promise<Result<undefined, Error>> // function to validate the extension's configuration
 * preDeployValidation?: (configuration: TConfiguration) => Promise<void> // function to validate the extension's configuration before deploying it
 * deployConfig?: (configuration: TConfiguration, directory: string) => Promise<{[key: string]: unknown}> // function to generate the extensions configuration payload to be deployed
 * shouldFetchCartUrl?: (configuration: TConfiguration) => boolean // function to determine if the extension should fetch the cart url
 * hasExtensionPointTarget?: (configuration: TConfiguration, target: string) => boolean // function to determine if the extension has a given extension point target
 * ```
 */
export function createExtensionSpecification(spec) {
    const defaults = {
        // these two fields are going to be overridden by the extension specification API response,
        // but we need them to have a default value for tests
        externalIdentifier: `${spec.identifier}_external`,
        additionalIdentifiers: [],
        externalName: capitalize(spec.identifier.replace(/_/g, ' ')),
        surface: 'test-surface',
        partnersWebIdentifier: spec.identifier,
        singleEntryPath: true,
        gated: false,
        schema: BaseSchema,
        registrationLimit: blocks.extensions.defaultRegistrationLimit,
        supportedFlavors: defaultExtensionFlavors,
    };
    return { ...defaults, ...spec };
}
//# sourceMappingURL=specification.js.map