/**
 * @remix-run/dev v1.19.3
 *
 * Copyright (c) Remix Software Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE.md file in the root directory of this source tree.
 *
 * @license MIT
 */
'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var node_child_process = require('node:child_process');
var path = require('node:path');
var node_url = require('node:url');
var fse = require('fs-extra');
var getPort = require('get-port');
var NPMCliPackageJson = require('@npmcli/package-json');
var semver = require('semver');
var routes = require('./config/routes.js');
var routesConvention = require('./config/routesConvention.js');
var serverModes = require('./config/serverModes.js');
var virtualModules = require('./compiler/server/virtualModules.js');
var flatRoutes = require('./config/flat-routes.js');
var detectPackageManager = require('./cli/detectPackageManager.js');
var logger = require('./tux/logger.js');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var path__default = /*#__PURE__*/_interopDefaultLegacy(path);
var fse__default = /*#__PURE__*/_interopDefaultLegacy(fse);
var getPort__default = /*#__PURE__*/_interopDefaultLegacy(getPort);
var NPMCliPackageJson__default = /*#__PURE__*/_interopDefaultLegacy(NPMCliPackageJson);

/**
 * The user-provided config in `remix.config.js`.
 */

/**
 * Fully resolved configuration object we use throughout Remix.
 */

/**
 * Returns a fully resolved config object from the remix.config.js in the given
 * root directory.
 */
async function readConfig(remixRoot, serverMode = serverModes.ServerMode.Production) {
  var _appConfig$future, _appConfig$future2, _appConfig$future3, _appConfig$future4, _appConfig$future5, _appConfig$future6, _appConfig$future7, _appConfig$future8, _appConfig$future9, _appConfig$future10, _appConfig$future11, _appConfig$future12, _appConfig$future13, _appConfig$future14, _appConfig$future15;
  if (!serverModes.isValidServerMode(serverMode)) {
    throw new Error(`Invalid server mode "${serverMode}"`);
  }
  if (!remixRoot) {
    remixRoot = process.env.REMIX_ROOT || process.cwd();
  }
  let rootDirectory = path__default["default"].resolve(remixRoot);
  let configFile = findConfig(rootDirectory, "remix.config", configExts);
  let appConfig = {};
  if (configFile) {
    let appConfigModule;
    try {
      var _appConfigModule;
      // shout out to next
      // https://github.com/vercel/next.js/blob/b15a976e11bf1dc867c241a4c1734757427d609c/packages/next/server/config.ts#L748-L765
      if (process.env.JEST_WORKER_ID) {
        // dynamic import does not currently work inside of vm which
        // jest relies on so we fall back to require for this case
        // https://github.com/nodejs/node/issues/35889
        appConfigModule = require(configFile);
      } else {
        appConfigModule = await import(node_url.pathToFileURL(configFile).href);
      }
      appConfig = ((_appConfigModule = appConfigModule) === null || _appConfigModule === void 0 ? void 0 : _appConfigModule.default) || appConfigModule;
    } catch (error) {
      throw new Error(`Error loading Remix config at ${configFile}\n${String(error)}`);
    }
  }
  if (appConfig.serverBuildTarget) {
    serverBuildTargetWarning();
  }
  if (!((_appConfig$future = appConfig.future) !== null && _appConfig$future !== void 0 && _appConfig$future.v2_errorBoundary)) {
    errorBoundaryWarning();
  }
  if (!((_appConfig$future2 = appConfig.future) !== null && _appConfig$future2 !== void 0 && _appConfig$future2.v2_normalizeFormMethod)) {
    formMethodWarning();
  }
  if (!((_appConfig$future3 = appConfig.future) !== null && _appConfig$future3 !== void 0 && _appConfig$future3.v2_meta)) {
    metaWarning();
  }
  if (!((_appConfig$future4 = appConfig.future) !== null && _appConfig$future4 !== void 0 && _appConfig$future4.v2_headers)) {
    headersWarning();
  }
  let isCloudflareRuntime = ["cloudflare-pages", "cloudflare-workers"].includes(appConfig.serverBuildTarget ?? "");
  let isDenoRuntime = appConfig.serverBuildTarget === "deno";
  let serverBuildPath = resolveServerBuildPath(rootDirectory, appConfig);
  let serverBuildTarget = appConfig.serverBuildTarget;
  let serverBuildTargetEntryModule = `export * from ${JSON.stringify(virtualModules.serverBuildVirtualModule.id)};`;
  let serverConditions = appConfig.serverConditions;
  let serverDependenciesToBundle = appConfig.serverDependenciesToBundle || [];
  let serverEntryPoint = appConfig.server;
  let serverMainFields = appConfig.serverMainFields;
  let serverMinify = appConfig.serverMinify;
  if (!appConfig.serverModuleFormat) {
    serverModuleFormatWarning();
  }
  let serverModuleFormat = appConfig.serverModuleFormat || "cjs";
  let serverPlatform = appConfig.serverPlatform || "node";
  if (isCloudflareRuntime) {
    serverConditions ?? (serverConditions = ["worker"]);
    serverDependenciesToBundle = "all";
    serverMainFields ?? (serverMainFields = ["browser", "module", "main"]);
    serverMinify ?? (serverMinify = true);
    serverModuleFormat = "esm";
    serverPlatform = "neutral";
  }
  if (isDenoRuntime) {
    serverConditions ?? (serverConditions = ["deno", "worker"]);
    serverDependenciesToBundle = "all";
    serverMainFields ?? (serverMainFields = ["module", "main"]);
    serverModuleFormat = "esm";
    serverPlatform = "neutral";
  }
  serverMainFields ?? (serverMainFields = serverModuleFormat === "esm" ? ["module", "main"] : ["main", "module"]);
  serverMinify ?? (serverMinify = false);
  let serverNodeBuiltinsPolyfill;
  if (appConfig.serverNodeBuiltinsPolyfill != null) {
    serverNodeBuiltinsPolyfill = appConfig.serverNodeBuiltinsPolyfill;
  } else if (serverPlatform !== "node") {
    serverNodeBuiltinsPolyfillWarning();
    serverNodeBuiltinsPolyfill = {
      modules: {
        // Note: Remove this in Remix v2
        // All polyfills are ultimately sourced from JSPM: https://github.com/jspm/jspm-core/tree/main/nodelibs/browser
        // Polyfills we choose to disable are explicitly configured here so we can note the reason for disabling them.
        // Links are provided here to make it easier to review the source code for each polyfill.
        _stream_duplex: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/_stream_duplex.js
        _stream_passthrough: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/_stream_passthrough.js
        _stream_readable: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/_stream_readable.js
        _stream_transform: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/_stream_transform.js
        _stream_writable: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/_stream_writable.js
        assert: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/assert.js
        "assert/strict": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/assert/strict.js
        async_hooks: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/async_hooks.js. Also Cloudflare Workers provides an implementation: https://developers.cloudflare.com/workers/runtime-apis/nodejs/asynclocalstorage/
        buffer: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/buffer.js
        child_process: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/child_process.js
        cluster: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/cluster.js
        console: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/console.js
        constants: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/constants.js
        crypto: "empty",
        // Polyfill exists (https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/crypto.js) but source code is over 2MB! Also, it was "empty" in esbuild-plugin-polyfill-node which we used previously as of Remix v1.17.0: https://github.com/cyco130/esbuild-plugin-polyfill-node/blob/9afcb6abaf9062a15daaffce9a14e478b365139c/src/index.ts#L144
        dgram: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/dgram.js
        diagnostics_channel: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/diagnostics_channel.js
        dns: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/dns.js
        "dns/promises": false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/dns/promises.js
        domain: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/domain.js
        events: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/events.js
        fs: "empty",
        // Polyfill was "empty" in esbuild-plugin-polyfill-node which we used previously as of Remix v1.17.0 (https://github.com/cyco130/esbuild-plugin-polyfill-node/blob/9afcb6abaf9062a15daaffce9a14e478b365139c/src/index.ts#L143C6-L143C6). Also, the polyfill immediately throws when importing in Cloudflare Workers due to top-level setTimeout usage which is not allowed outside of the request lifecycle: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/fs.js
        "fs/promises": "empty",
        // See above
        http: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/http.js
        http2: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/http2.js
        https: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/https.js
        module: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/module.js
        net: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/net.js
        os: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/os.js
        path: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/path.js
        "path/posix": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/path/posix.js
        "path/win32": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/path/win32.js
        perf_hooks: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/perf_hooks.js
        process: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/process.js
        punycode: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/punycode.js
        querystring: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/querystring.js
        readline: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/readline.js
        repl: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/repl.js
        stream: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/stream.js
        "stream/promises": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/stream/promises.js
        "stream/web": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/stream/web.js
        string_decoder: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/string_decoder.js
        sys: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/sys.js
        timers: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/timers.js
        "timers/promises": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/timers/promises.js
        tls: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/tls.js
        tty: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/tty.js - Effectively not implemented, but provides `isatty` as `false` so consumers can check to avoid it
        url: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/url.js
        util: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/util.js
        "util/types": true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/util/types.js
        v8: false,
        // Unimplemented, throws on usage: https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/v8.js
        vm: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/vm.js
        wasi: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/wasi.js
        worker_threads: true,
        // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/worker_threads.js
        zlib: true // https://github.com/jspm/jspm-core/blob/main/nodelibs/browser/zlib.js
      }
    };
  }

  if (appConfig.future) {
    if ("unstable_cssModules" in appConfig.future) {
      logger.logger.warn("The `future.unstable_cssModules` config option has been removed", {
        details: ["CSS Modules are now enabled automatically.", "You should remove the `unstable_cssModules` option from your Remix config."],
        key: "unstable_cssModules"
      });
    }
    if ("unstable_cssSideEffectImports" in appConfig.future) {
      logger.logger.warn("The `future.unstable_cssSideEffectImports` config option has been removed", {
        details: ["CSS side-effect imports are now enabled automatically.", "You should remove the `unstable_cssSideEffectImports` option from your Remix config"],
        key: "unstable_cssSideEffectImports"
      });
    }
    if ("unstable_vanillaExtract" in appConfig.future) {
      logger.logger.warn("The `future.unstable_vanillaExtract` config option has been removed.", {
        details: ["Vanilla Extract is now enabled automatically.", "You should remove the `unstable_vanillaExtract` option from your Remix config"],
        key: "unstable_vanillaExtract"
      });
    }
    if (appConfig.future.unstable_postcss !== undefined) {
      logger.logger.warn("The `future.unstable_postcss` config option has been deprecated.", {
        details: ["PostCSS support is now stable.", "Use the `postcss` config option instead."],
        key: "unstable_postcss"
      });
    }
    if (appConfig.future.unstable_tailwind !== undefined) {
      logger.logger.warn("The `future.unstable_tailwind` config option has been deprecated.", {
        details: ["Tailwind support is now stable.", "Use the `tailwind` config option instead."],
        key: "unstable_tailwind"
      });
    }
    if ("unstable_dev" in appConfig.future) {
      logger.logger.warn("The `future.unstable_dev` config option has been removed", {
        details: ["The v2 dev server is now stable.", "Use the `future.v2_dev` config option instead.", "-> https://remix.run/docs/en/main/pages/v2#dev-server"],
        key: "unstable_dev"
      });
    }
  }
  let mdx = appConfig.mdx;
  let postcss = appConfig.postcss ?? ((_appConfig$future5 = appConfig.future) === null || _appConfig$future5 === void 0 ? void 0 : _appConfig$future5.unstable_postcss) === true;
  let tailwind = appConfig.tailwind ?? ((_appConfig$future6 = appConfig.future) === null || _appConfig$future6 === void 0 ? void 0 : _appConfig$future6.unstable_tailwind) === true;
  let appDirectory = path__default["default"].resolve(rootDirectory, appConfig.appDirectory || "app");
  let cacheDirectory = path__default["default"].resolve(rootDirectory, appConfig.cacheDirectory || ".cache");
  let defaultsDirectory = path__default["default"].resolve(__dirname, "config", "defaults");
  let userEntryClientFile = findEntry(appDirectory, "entry.client");
  let userEntryServerFile = findEntry(appDirectory, "entry.server");
  let entryServerFile;
  let entryClientFile;
  let pkgJson = await NPMCliPackageJson__default["default"].load(remixRoot);
  let deps = pkgJson.content.dependencies ?? {};
  if (userEntryServerFile) {
    entryServerFile = userEntryServerFile;
  } else {
    let serverRuntime = deps["@remix-run/deno"] ? "deno" : deps["@remix-run/cloudflare"] ? "cloudflare" : deps["@remix-run/node"] ? "node" : undefined;
    if (!serverRuntime) {
      let serverRuntimes = ["@remix-run/deno", "@remix-run/cloudflare", "@remix-run/node"];
      let formattedList = disjunctionListFormat.format(serverRuntimes);
      throw new Error(`Could not determine server runtime. Please install one of the following: ${formattedList}`);
    }
    let clientRenderer = deps["@remix-run/react"] ? "react" : undefined;
    if (!clientRenderer) {
      throw new Error(`Could not determine renderer. Please install the following: @remix-run/react`);
    }
    let maybeReactVersion = semver.coerce(deps.react);
    if (!maybeReactVersion) {
      let react = ["react", "react-dom"];
      let list = conjunctionListFormat.format(react);
      throw new Error(`Could not determine React version. Please install the following packages: ${list}`);
    }
    let type = maybeReactVersion.major >= 18 || maybeReactVersion.raw === "0.0.0" ? "stream" : "string";
    if (!deps["isbot"] && type === "stream") {
      console.log("adding `isbot` to your package.json, you should commit this change");
      pkgJson.update({
        dependencies: {
          ...pkgJson.content.dependencies,
          isbot: "latest"
        }
      });
      await pkgJson.save();
      let packageManager = detectPackageManager.detectPackageManager() ?? "npm";
      node_child_process.execSync(`${packageManager} install`, {
        cwd: remixRoot,
        stdio: "inherit"
      });
    }
    entryServerFile = `${serverRuntime}/entry.server.${clientRenderer}-${type}.tsx`;
  }
  if (userEntryClientFile) {
    entryClientFile = userEntryClientFile;
  } else {
    let clientRenderer = deps["@remix-run/react"] ? "react" : undefined;
    if (!clientRenderer) {
      throw new Error(`Could not determine runtime. Please install the following: @remix-run/react`);
    }
    let maybeReactVersion = semver.coerce(deps.react);
    if (!maybeReactVersion) {
      let react = ["react", "react-dom"];
      let list = conjunctionListFormat.format(react);
      throw new Error(`Could not determine React version. Please install the following packages: ${list}`);
    }
    let type = maybeReactVersion.major >= 18 || maybeReactVersion.raw === "0.0.0" ? "stream" : "string";
    entryClientFile = `entry.client.${clientRenderer}-${type}.tsx`;
  }
  let entryClientFilePath = userEntryClientFile ? path__default["default"].resolve(appDirectory, userEntryClientFile) : path__default["default"].resolve(defaultsDirectory, entryClientFile);
  let entryServerFilePath = userEntryServerFile ? path__default["default"].resolve(appDirectory, userEntryServerFile) : path__default["default"].resolve(defaultsDirectory, entryServerFile);
  if (appConfig.browserBuildDirectory) {
    browserBuildDirectoryWarning();
  }
  let assetsBuildDirectory = appConfig.assetsBuildDirectory || appConfig.browserBuildDirectory || path__default["default"].join("public", "build");
  let absoluteAssetsBuildDirectory = path__default["default"].resolve(rootDirectory, assetsBuildDirectory);
  if (appConfig.devServerPort) {
    devServerPortWarning();
  }
  if (appConfig.devServerBroadcastDelay) {
    devServerBroadcastDelayWarning();
  }
  let devServerPort = Number(process.env.REMIX_DEV_SERVER_WS_PORT) || (await getPort__default["default"]({
    port: Number(appConfig.devServerPort) || 8002
  }));
  // set env variable so un-bundled servers can use it
  process.env.REMIX_DEV_SERVER_WS_PORT = String(devServerPort);
  let devServerBroadcastDelay = appConfig.devServerBroadcastDelay || 0;
  let defaultPublicPath = appConfig.serverBuildTarget === "arc" ? "/_static/build/" : "/build/";
  let publicPath = addTrailingSlash(appConfig.publicPath || defaultPublicPath);
  let rootRouteFile = findEntry(appDirectory, "root");
  if (!rootRouteFile) {
    throw new Error(`Missing "root" route file in ${appDirectory}`);
  }
  let routes$1 = {
    root: {
      path: "",
      id: "root",
      file: rootRouteFile
    }
  };
  let routesConvention$1;
  if ((_appConfig$future7 = appConfig.future) !== null && _appConfig$future7 !== void 0 && _appConfig$future7.v2_routeConvention) {
    routesConvention$1 = flatRoutes.flatRoutes;
  } else {
    flatRoutesWarning();
    routesConvention$1 = routesConvention.defineConventionalRoutes;
  }
  if (fse__default["default"].existsSync(path__default["default"].resolve(appDirectory, "routes"))) {
    let conventionalRoutes = routesConvention$1(appDirectory, appConfig.ignoredRouteFiles);
    for (let route of Object.values(conventionalRoutes)) {
      routes$1[route.id] = {
        ...route,
        parentId: route.parentId || "root"
      };
    }
  }
  if (appConfig.routes) {
    let manualRoutes = await appConfig.routes(routes.defineRoutes);
    for (let route of Object.values(manualRoutes)) {
      routes$1[route.id] = {
        ...route,
        parentId: route.parentId || "root"
      };
    }
  }
  let watchPaths = [];
  if (typeof appConfig.watchPaths === "function") {
    let directories = await appConfig.watchPaths();
    watchPaths = watchPaths.concat(Array.isArray(directories) ? directories : [directories]);
  } else if (appConfig.watchPaths) {
    watchPaths = watchPaths.concat(Array.isArray(appConfig.watchPaths) ? appConfig.watchPaths : [appConfig.watchPaths]);
  }

  // When tsconfigPath is undefined, the default "tsconfig.json" is not
  // found in the root directory.
  let tsconfigPath;
  let rootTsconfig = path__default["default"].resolve(rootDirectory, "tsconfig.json");
  let rootJsConfig = path__default["default"].resolve(rootDirectory, "jsconfig.json");
  if (fse__default["default"].existsSync(rootTsconfig)) {
    tsconfigPath = rootTsconfig;
  } else if (fse__default["default"].existsSync(rootJsConfig)) {
    tsconfigPath = rootJsConfig;
  }
  let future = {
    v2_dev: ((_appConfig$future8 = appConfig.future) === null || _appConfig$future8 === void 0 ? void 0 : _appConfig$future8.v2_dev) ?? false,
    unstable_postcss: ((_appConfig$future9 = appConfig.future) === null || _appConfig$future9 === void 0 ? void 0 : _appConfig$future9.unstable_postcss) === true,
    unstable_tailwind: ((_appConfig$future10 = appConfig.future) === null || _appConfig$future10 === void 0 ? void 0 : _appConfig$future10.unstable_tailwind) === true,
    v2_errorBoundary: ((_appConfig$future11 = appConfig.future) === null || _appConfig$future11 === void 0 ? void 0 : _appConfig$future11.v2_errorBoundary) === true,
    v2_headers: ((_appConfig$future12 = appConfig.future) === null || _appConfig$future12 === void 0 ? void 0 : _appConfig$future12.v2_headers) === true,
    v2_meta: ((_appConfig$future13 = appConfig.future) === null || _appConfig$future13 === void 0 ? void 0 : _appConfig$future13.v2_meta) === true,
    v2_normalizeFormMethod: ((_appConfig$future14 = appConfig.future) === null || _appConfig$future14 === void 0 ? void 0 : _appConfig$future14.v2_normalizeFormMethod) === true,
    v2_routeConvention: ((_appConfig$future15 = appConfig.future) === null || _appConfig$future15 === void 0 ? void 0 : _appConfig$future15.v2_routeConvention) === true
  };
  return {
    appDirectory,
    cacheDirectory,
    entryClientFile,
    entryClientFilePath,
    entryServerFile,
    entryServerFilePath,
    devServerPort,
    devServerBroadcastDelay,
    assetsBuildDirectory: absoluteAssetsBuildDirectory,
    relativeAssetsBuildDirectory: assetsBuildDirectory,
    publicPath,
    rootDirectory,
    routes: routes$1,
    serverBuildPath,
    serverBuildTarget,
    serverBuildTargetEntryModule,
    serverConditions,
    serverDependenciesToBundle,
    serverEntryPoint,
    serverMainFields,
    serverMinify,
    serverMode,
    serverModuleFormat,
    serverNodeBuiltinsPolyfill,
    serverPlatform,
    mdx,
    postcss,
    tailwind,
    watchPaths,
    tsconfigPath,
    future
  };
}
function addTrailingSlash(path) {
  return path.endsWith("/") ? path : path + "/";
}
const entryExts = [".js", ".jsx", ".ts", ".tsx"];
function findEntry(dir, basename) {
  for (let ext of entryExts) {
    let file = path__default["default"].resolve(dir, basename + ext);
    if (fse__default["default"].existsSync(file)) return path__default["default"].relative(dir, file);
  }
  return undefined;
}
const configExts = [".js", ".cjs", ".mjs"];
function findConfig(dir, basename, extensions) {
  for (let ext of extensions) {
    let name = basename + ext;
    let file = path__default["default"].join(dir, name);
    if (fse__default["default"].existsSync(file)) return file;
  }
  return undefined;
}
const resolveServerBuildPath = (rootDirectory, appConfig) => {
  let serverBuildPath = "build/index.js";
  switch (appConfig.serverBuildTarget) {
    case "arc":
      serverBuildPath = "server/index.js";
      break;
    case "cloudflare-pages":
      serverBuildPath = "functions/[[path]].js";
      break;
    case "netlify":
      serverBuildPath = ".netlify/functions-internal/server.js";
      break;
    case "vercel":
      serverBuildPath = "api/index.js";
      break;
  }

  // retain deprecated behavior for now
  if (appConfig.serverBuildDirectory) {
    serverBuildDirectoryWarning();
    serverBuildPath = path__default["default"].join(appConfig.serverBuildDirectory, "index.js");
  }
  if (appConfig.serverBuildPath) {
    serverBuildPath = appConfig.serverBuildPath;
  }
  return path__default["default"].resolve(rootDirectory, serverBuildPath);
};

// adds types for `Intl.ListFormat` to the global namespace
// we could also update our `tsconfig.json` to include `lib: ["es2021"]`
let conjunctionListFormat = new Intl.ListFormat("en", {
  style: "long",
  type: "conjunction"
});
let disjunctionListFormat = new Intl.ListFormat("en", {
  style: "long",
  type: "disjunction"
});
let browserBuildDirectoryWarning = () => logger.logger.warn("The `browserBuildDirectory` config option will be removed in v2", {
  details: ["You can use the `assetsBuildDirectory` config option instead.", "-> https://remix.run/docs/en/v1.15.0/pages/v2#browserbuilddirectory"],
  key: "browserBuildDirectoryWarning"
});
let devServerBroadcastDelayWarning = () => logger.logger.warn("The `devServerBroadcastDelay` config option will be removed in v2", {
  details: ["Enable `v2_dev` to eliminate the race conditions that necessitated this option.", "-> https://remix.run/docs/en/v1.19.3/pages/v2#devserverbroadcastdelay"],
  key: "devServerBroadcastDelayWarning"
});
let devServerPortWarning = () => logger.logger.warn("The `devServerPort` config option will be removed in v2", {
  details: ["Enable `v2_dev` and use `--port` / `v2_dev.port` option instead.", "-> https://remix.run/docs/en/v1.19.3/pages/v2#devserverport"],
  key: "devServerPortWarning"
});
let serverBuildDirectoryWarning = () => logger.logger.warn("The `serverBuildDirectory` config option will be removed in v2", {
  details: ["You can use the `serverBuildPath` config option instead.", "-> https://remix.run/docs/en/v1.15.0/pages/v2#serverbuilddirectory"],
  key: "serverBuildDirectoryWarning"
});
let serverBuildTargetWarning = () => logger.logger.warn("The `serverBuildTarget` config option will be removed in v2", {
  details: ["You can specify multiple server module config options instead to achieve the same result.", "-> https://remix.run/docs/en/v1.15.0/pages/v2#serverbuildtarget"],
  key: "serverBuildTargetWarning"
});
let serverModuleFormatWarning = () => logger.logger.warn("The default server module format is changing in v2", {
  details: ["The default format will change from `cjs` to `esm`.", "You can keep using `cjs` by explicitly specifying `serverModuleFormat: 'cjs'`.", "You can opt-in early to this change by explicitly specifying `serverModuleFormat: 'esm'`", "-> https://remix.run/docs/en/v1.16.0/pages/v2#servermoduleformat"],
  key: "serverModuleFormatWarning"
});
let serverNodeBuiltinsPolyfillWarning = () => logger.logger.warn("The `serverNodeBuiltinsPolyfill` config default option will be changing in v2", {
  details: ["Server polyfills will no longer be provided by default for non-Node.js platforms.", "You can prepare for this change by specifying server polyfills, or opting out entirely.", "-> https://remix.run/docs/en/v1.19.0/pages/v2#servernodebuiltinspolyfill"],
  key: "serverNodeBuiltinsPolyfillWarning"
});
let futureFlagWarning = args => () => {
  logger.logger.warn(args.message, {
    key: args.flag,
    details: [`You can use the \`${args.flag}\` future flag to opt-in early.`, `-> ${args.link}`]
  });
};
let flatRoutesWarning = futureFlagWarning({
  message: "The route file convention is changing in v2",
  flag: "v2_routeConvention",
  link: "https://remix.run/docs/en/v1.15.0/pages/v2#file-system-route-convention"
});
let errorBoundaryWarning = futureFlagWarning({
  message: "The `CatchBoundary` and `ErrorBoundary` API is changing in v2",
  flag: "v2_errorBoundary",
  link: "https://remix.run/docs/en/v1.15.0/pages/v2#catchboundary-and-errorboundary"
});
let formMethodWarning = futureFlagWarning({
  message: "The `formMethod` API is changing in v2",
  flag: "v2_normalizeFormMethod",
  link: "https://remix.run/docs/en/v1.15.0/pages/v2#formMethod"
});
let metaWarning = futureFlagWarning({
  message: "The route `meta` API is changing in v2",
  flag: "v2_meta",
  link: "https://remix.run/docs/en/v1.15.0/pages/v2#meta"
});
let headersWarning = futureFlagWarning({
  message: "The route `headers` API is changing in v2",
  flag: "v2_headers",
  link: "https://remix.run/docs/en/v1.17.0/pages/v2#route-headers"
});

exports.findConfig = findConfig;
exports.readConfig = readConfig;
