import { createReadStream } from 'fs';
import { request as request$1 } from 'http';
import { request } from 'https';
import { Readable } from 'stream';
import { fileURLToPath } from 'url';
import { Blob } from 'buffer';
import { EventTarget, CustomEvent } from '@whatwg-node/events';
import busboy from 'busboy';

// Will be removed after v14 reaches EOL
class PonyfillAbortError extends Error {
    constructor(reason) {
        let message = 'The operation was aborted.';
        if (reason) {
            message += ` reason: ${reason}`;
        }
        super(message, {
            cause: reason,
        });
        this.name = 'AbortError';
    }
    get reason() {
        return this.cause;
    }
}

function createController(desiredSize, readable) {
    let chunks = [];
    let _closed = false;
    return {
        desiredSize,
        enqueue(chunk) {
            chunks.push(Buffer.from(chunk));
        },
        close() {
            if (chunks.length > 0) {
                this._flush();
            }
            readable.push(null);
            _closed = true;
        },
        error(error) {
            if (chunks.length > 0) {
                this._flush();
            }
            readable.destroy(error);
        },
        get _closed() {
            return _closed;
        },
        _flush() {
            if (chunks.length > 0) {
                const concatenated = Buffer.concat(chunks);
                readable.push(concatenated);
                chunks = [];
            }
        },
    };
}
class PonyfillReadableStream {
    constructor(underlyingSource) {
        this.locked = false;
        if (underlyingSource instanceof PonyfillReadableStream) {
            this.readable = underlyingSource.readable;
        }
        else if (underlyingSource && 'read' in underlyingSource) {
            this.readable = underlyingSource;
        }
        else if (underlyingSource && 'getReader' in underlyingSource) {
            let reader;
            let started = false;
            this.readable = new Readable({
                read() {
                    if (!started) {
                        started = true;
                        reader = underlyingSource.getReader();
                    }
                    reader
                        .read()
                        .then(({ value, done }) => {
                        if (done) {
                            this.push(null);
                        }
                        else {
                            this.push(value);
                        }
                    })
                        .catch(err => {
                        this.destroy(err);
                    });
                },
                destroy(err, callback) {
                    reader.cancel(err).then(() => callback(err), callback);
                },
            });
        }
        else {
            let started = false;
            this.readable = new Readable({
                async read(desiredSize) {
                    var _a, _b;
                    const controller = createController(desiredSize, this);
                    if (!started) {
                        started = true;
                        await ((_a = underlyingSource === null || underlyingSource === void 0 ? void 0 : underlyingSource.start) === null || _a === void 0 ? void 0 : _a.call(underlyingSource, controller));
                    }
                    if (!controller._closed) {
                        await ((_b = underlyingSource === null || underlyingSource === void 0 ? void 0 : underlyingSource.pull) === null || _b === void 0 ? void 0 : _b.call(underlyingSource, controller));
                    }
                    controller._flush();
                },
                async destroy(err, callback) {
                    var _a;
                    try {
                        await ((_a = underlyingSource === null || underlyingSource === void 0 ? void 0 : underlyingSource.cancel) === null || _a === void 0 ? void 0 : _a.call(underlyingSource, err));
                        callback(null);
                    }
                    catch (err) {
                        callback(err);
                    }
                },
            });
        }
    }
    cancel(reason) {
        this.readable.destroy(reason);
        return Promise.resolve();
    }
    getReader(_options) {
        const iterator = this.readable[Symbol.asyncIterator]();
        this.locked = true;
        return {
            read() {
                return iterator.next();
            },
            releaseLock: () => {
                var _a;
                (_a = iterator.return) === null || _a === void 0 ? void 0 : _a.call(iterator);
                this.locked = false;
            },
            cancel: async (reason) => {
                var _a;
                await ((_a = iterator.return) === null || _a === void 0 ? void 0 : _a.call(iterator, reason));
                this.locked = false;
            },
            closed: new Promise((resolve, reject) => {
                this.readable.once('end', resolve);
                this.readable.once('error', reject);
            }),
        };
    }
    [Symbol.asyncIterator]() {
        return this.readable[Symbol.asyncIterator]();
    }
    tee() {
        throw new Error('Not implemented');
    }
    async pipeTo(destination) {
        const writer = destination.getWriter();
        await writer.ready;
        for await (const chunk of this.readable) {
            await writer.write(chunk);
        }
        await writer.ready;
        return writer.close();
    }
    pipeThrough({ writable, readable, }) {
        this.pipeTo(writable);
        return readable;
    }
    static [Symbol.hasInstance](instance) {
        return instance != null && typeof instance === 'object' && 'getReader' in instance;
    }
}

class DummyBlob {
    constructor() {
        throw new Error('Blob is not supported in this environment, if you are using an older version of Node v14, please upgrade to v14.17.0 or higher');
    }
}
// Will be removed after v14 reaches EOL
// Needed because v14 doesn't have .stream() implemented
class PonyfillBlob extends (Blob || DummyBlob) {
    stream() {
        return new PonyfillReadableStream({
            start: async (controller) => {
                const arrayBuffer = await this.arrayBuffer();
                const buffer = Buffer.from(arrayBuffer);
                controller.enqueue(buffer);
                controller.close();
            },
        });
    }
    slice(...args) {
        return super.slice(...args);
    }
}

// Will be removed after v14 reaches EOL
class PonyfillAbortSignal extends EventTarget {
    constructor() {
        super(...arguments);
        this.aborted = false;
        this._onabort = null;
    }
    throwIfAborted() {
        if (this.aborted) {
            throw new PonyfillAbortError();
        }
    }
    get onabort() {
        return this._onabort;
    }
    set onabort(value) {
        if (this._onabort) {
            this.removeEventListener('abort', this._onabort);
        }
        this.addEventListener('abort', value);
    }
    abort(reason) {
        const abortEvent = new CustomEvent('abort', { detail: reason });
        this.dispatchEvent(abortEvent);
    }
    static timeout(milliseconds) {
        const signal = new PonyfillAbortSignal();
        setTimeout(() => signal.abort(`timeout in ${milliseconds} ms`), milliseconds);
        return signal;
    }
}

// Will be removed after v14 reaches EOL
class PonyfillAbortController {
    constructor() {
        this.signal = new PonyfillAbortSignal();
    }
    abort(reason) {
        this.signal.abort(reason);
    }
}

class PonyfillFile extends PonyfillBlob {
    constructor(fileBits, name, options) {
        super(fileBits, options);
        this.name = name;
        this.webkitRelativePath = '';
        this.lastModified = (options === null || options === void 0 ? void 0 : options.lastModified) || Date.now();
    }
}

class PonyfillFormData {
    constructor() {
        this.map = new Map();
    }
    append(name, value, fileName) {
        let values = this.map.get(name);
        if (!values) {
            values = [];
            this.map.set(name, values);
        }
        const entry = isBlob(value)
            ? getNormalizedFile(name, value, fileName)
            : value;
        values.push(entry);
    }
    delete(name) {
        this.map.delete(name);
    }
    get(name) {
        const values = this.map.get(name);
        return values ? values[0] : null;
    }
    getAll(name) {
        return this.map.get(name) || [];
    }
    has(name) {
        return this.map.has(name);
    }
    set(name, value, fileName) {
        const entry = isBlob(value)
            ? getNormalizedFile(name, value, fileName)
            : value;
        this.map.set(name, [entry]);
    }
    *[Symbol.iterator]() {
        for (const [key, values] of this.map) {
            for (const value of values) {
                yield [key, value];
            }
        }
    }
    forEach(callback) {
        for (const [key, value] of this) {
            callback(value, key, this);
        }
    }
}
function getStreamFromFormData(formData, boundary = '---') {
    const entries = [];
    let sentInitialHeader = false;
    return new PonyfillReadableStream({
        start: controller => {
            formData.forEach((value, key) => {
                if (!sentInitialHeader) {
                    controller.enqueue(Buffer.from(`--${boundary}\r\n`));
                    sentInitialHeader = true;
                }
                entries.push([key, value]);
            });
            if (!sentInitialHeader) {
                controller.enqueue(Buffer.from(`--${boundary}--\r\n`));
                controller.close();
            }
        },
        pull: async (controller) => {
            const entry = entries.shift();
            if (entry) {
                const [key, value] = entry;
                if (typeof value === 'string') {
                    controller.enqueue(Buffer.from(`Content-Disposition: form-data; name="${key}"\r\n\r\n`));
                    controller.enqueue(Buffer.from(value));
                }
                else {
                    let filenamePart = '';
                    if (value.name) {
                        filenamePart = `; filename="${value.name}"`;
                    }
                    controller.enqueue(Buffer.from(`Content-Disposition: form-data; name="${key}"${filenamePart}\r\n`));
                    controller.enqueue(Buffer.from(`Content-Type: ${value.type || 'application/octet-stream'}\r\n\r\n`));
                    const entryStream = value.stream();
                    for await (const chunk of entryStream) {
                        controller.enqueue(chunk);
                    }
                }
                if (entries.length === 0) {
                    controller.enqueue(Buffer.from(`\r\n--${boundary}--\r\n`));
                    controller.close();
                }
                else {
                    controller.enqueue(Buffer.from(`\r\n--${boundary}\r\n`));
                }
            }
            else {
                controller.enqueue(Buffer.from(`\r\n--${boundary}--\r\n`));
                controller.close();
            }
        },
    });
}
function getNormalizedFile(name, blob, fileName) {
    if (blob instanceof PonyfillFile) {
        if (fileName != null) {
            return new PonyfillFile([blob], fileName, {
                type: blob.type,
                lastModified: blob.lastModified,
            });
        }
        return blob;
    }
    return new PonyfillFile([blob], fileName || name, { type: blob.type });
}
function isBlob(value) {
    return value != null && typeof value === 'object' && typeof value.arrayBuffer === 'function';
}

var BodyInitType;
(function (BodyInitType) {
    BodyInitType["ReadableStream"] = "ReadableStream";
    BodyInitType["Blob"] = "Blob";
    BodyInitType["FormData"] = "FormData";
    BodyInitType["ArrayBuffer"] = "ArrayBuffer";
    BodyInitType["String"] = "String";
    BodyInitType["Readable"] = "Readable";
    BodyInitType["Buffer"] = "Buffer";
    BodyInitType["Uint8Array"] = "Uint8Array";
})(BodyInitType || (BodyInitType = {}));
class PonyfillBody {
    constructor(bodyInit, options = {}) {
        this.bodyInit = bodyInit;
        this.options = options;
        this.bodyUsed = false;
        this.contentType = null;
        this.contentLength = null;
        this._bodyFactory = () => null;
        this._generatedBody = null;
        const { bodyFactory, contentType, contentLength, bodyType } = processBodyInit(bodyInit);
        this._bodyFactory = bodyFactory;
        this.contentType = contentType;
        this.contentLength = contentLength;
        this.bodyType = bodyType;
    }
    generateBody() {
        if (this._generatedBody) {
            return this._generatedBody;
        }
        const body = this._bodyFactory();
        this._generatedBody = body;
        return body;
    }
    get body() {
        const _body = this.generateBody();
        if (_body != null) {
            const ponyfillReadableStream = _body;
            const readable = _body.readable;
            return new Proxy(_body.readable, {
                get(_, prop) {
                    if (prop in ponyfillReadableStream) {
                        const ponyfillReadableStreamProp = ponyfillReadableStream[prop];
                        if (typeof ponyfillReadableStreamProp === 'function') {
                            return ponyfillReadableStreamProp.bind(ponyfillReadableStream);
                        }
                        return ponyfillReadableStreamProp;
                    }
                    if (prop in readable) {
                        const readableProp = readable[prop];
                        if (typeof readableProp === 'function') {
                            return readableProp.bind(readable);
                        }
                        return readableProp;
                    }
                },
            });
        }
        return null;
    }
    async arrayBuffer() {
        if (this.bodyType === BodyInitType.ArrayBuffer) {
            return this.bodyInit;
        }
        if (this.bodyType === BodyInitType.Uint8Array || this.bodyType === BodyInitType.Buffer) {
            return this.bodyInit.buffer;
        }
        const blob = await this.blob();
        return blob.arrayBuffer();
    }
    async blob() {
        if (this.bodyType === BodyInitType.Blob) {
            return this.bodyInit;
        }
        if (this.bodyType === BodyInitType.String ||
            this.bodyType === BodyInitType.Buffer ||
            this.bodyType === BodyInitType.Uint8Array) {
            const bodyInitTyped = this.bodyInit;
            return new PonyfillBlob([bodyInitTyped], {
                type: this.contentType || '',
            });
        }
        if (this.bodyType === BodyInitType.ArrayBuffer) {
            const bodyInitTyped = this.bodyInit;
            const buf = Buffer.from(bodyInitTyped, undefined, bodyInitTyped.byteLength);
            return new PonyfillBlob([buf], {
                type: this.contentType || '',
            });
        }
        const chunks = [];
        const _body = this.generateBody();
        if (_body) {
            for await (const chunk of _body.readable) {
                chunks.push(chunk);
            }
        }
        return new PonyfillBlob(chunks, {
            type: this.contentType || '',
        });
    }
    formData(opts) {
        if (this.bodyType === BodyInitType.FormData) {
            return Promise.resolve(this.bodyInit);
        }
        const formData = new PonyfillFormData();
        const _body = this.generateBody();
        if (_body == null) {
            return Promise.resolve(formData);
        }
        const formDataLimits = {
            ...this.options.formDataLimits,
            ...opts === null || opts === void 0 ? void 0 : opts.formDataLimits,
        };
        return new Promise((resolve, reject) => {
            const bb = busboy({
                headers: {
                    'content-type': this.contentType || '',
                },
                limits: formDataLimits,
                defParamCharset: 'utf-8',
            });
            bb.on('field', (name, value, { nameTruncated, valueTruncated }) => {
                if (nameTruncated) {
                    reject(new Error(`Field name size exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.fieldNameSize} bytes`));
                }
                if (valueTruncated) {
                    reject(new Error(`Field value size exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.fieldSize} bytes`));
                }
                formData.set(name, value);
            });
            bb.on('fieldsLimit', () => {
                reject(new Error(`Fields limit exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.fields}`));
            });
            bb.on('file', (name, fileStream, { filename, mimeType }) => {
                const chunks = [];
                fileStream.on('limit', () => {
                    reject(new Error(`File size limit exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.fileSize} bytes`));
                });
                fileStream.on('data', chunk => {
                    chunks.push(Buffer.from(chunk));
                });
                fileStream.on('close', () => {
                    if (fileStream.truncated) {
                        reject(new Error(`File size limit exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.fileSize} bytes`));
                    }
                    const file = new PonyfillFile(chunks, filename, { type: mimeType });
                    formData.set(name, file);
                });
            });
            bb.on('filesLimit', () => {
                reject(new Error(`Files limit exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.files}`));
            });
            bb.on('partsLimit', () => {
                reject(new Error(`Parts limit exceeded: ${formDataLimits === null || formDataLimits === void 0 ? void 0 : formDataLimits.parts}`));
            });
            bb.on('close', () => {
                resolve(formData);
            });
            bb.on('error', err => {
                reject(err);
            });
            _body === null || _body === void 0 ? void 0 : _body.readable.pipe(bb);
        });
    }
    async buffer() {
        if (this.bodyType === BodyInitType.Buffer) {
            return this.bodyInit;
        }
        if (this.bodyType === BodyInitType.Uint8Array || this.bodyType === BodyInitType.ArrayBuffer) {
            const bodyInitTyped = this.bodyInit;
            const buffer = Buffer.from(bodyInitTyped, 'byteOffset' in bodyInitTyped ? bodyInitTyped.byteOffset : undefined, bodyInitTyped.byteLength);
            return buffer;
        }
        const blob = await this.blob();
        const arrayBuffer = await blob.arrayBuffer();
        return Buffer.from(arrayBuffer, undefined, arrayBuffer.byteLength);
    }
    async json() {
        const text = await this.text();
        return JSON.parse(text);
    }
    async text() {
        if (this.bodyType === BodyInitType.String) {
            return this.bodyInit;
        }
        if (this.bodyType === BodyInitType.Buffer) {
            return this.bodyInit.toString('utf-8');
        }
        if (this.bodyType === BodyInitType.ArrayBuffer || this.bodyType === BodyInitType.Uint8Array) {
            const bodyInitTyped = this.bodyInit;
            const buffer = Buffer.from(bodyInitTyped, 'byteOffset' in bodyInitTyped ? bodyInitTyped.byteOffset : undefined, bodyInitTyped.byteLength);
            return buffer.toString('utf-8');
        }
        const blob = await this.blob();
        return blob.text();
    }
}
function processBodyInit(bodyInit) {
    if (bodyInit == null) {
        return {
            bodyFactory: () => null,
            contentType: null,
            contentLength: null,
        };
    }
    if (typeof bodyInit === 'string') {
        return {
            bodyType: BodyInitType.String,
            contentType: 'text/plain;charset=UTF-8',
            contentLength: Buffer.byteLength(bodyInit),
            bodyFactory() {
                const readable = Readable.from(bodyInit);
                return new PonyfillReadableStream(readable);
            },
        };
    }
    if (bodyInit instanceof PonyfillReadableStream) {
        return {
            bodyType: BodyInitType.ReadableStream,
            bodyFactory: () => bodyInit,
            contentType: null,
            contentLength: null,
        };
    }
    if (bodyInit instanceof PonyfillBlob) {
        return {
            bodyType: BodyInitType.Blob,
            contentType: bodyInit.type,
            contentLength: bodyInit.size,
            bodyFactory() {
                return bodyInit.stream();
            },
        };
    }
    if (bodyInit instanceof Buffer) {
        const contentLength = bodyInit.length;
        return {
            bodyType: BodyInitType.Buffer,
            contentLength,
            contentType: null,
            bodyFactory() {
                const readable = Readable.from(bodyInit);
                const body = new PonyfillReadableStream(readable);
                return body;
            },
        };
    }
    if (bodyInit instanceof Uint8Array) {
        const contentLength = bodyInit.byteLength;
        return {
            bodyType: BodyInitType.Uint8Array,
            contentLength,
            contentType: null,
            bodyFactory() {
                const readable = Readable.from(bodyInit);
                const body = new PonyfillReadableStream(readable);
                return body;
            },
        };
    }
    if ('buffer' in bodyInit) {
        const contentLength = bodyInit.byteLength;
        return {
            contentLength,
            contentType: null,
            bodyFactory() {
                const buffer = Buffer.from(bodyInit.buffer, bodyInit.byteOffset, bodyInit.byteLength);
                const readable = Readable.from(buffer);
                const body = new PonyfillReadableStream(readable);
                return body;
            },
        };
    }
    if (bodyInit instanceof ArrayBuffer) {
        const contentLength = bodyInit.byteLength;
        return {
            bodyType: BodyInitType.ArrayBuffer,
            contentType: null,
            contentLength,
            bodyFactory() {
                const buffer = Buffer.from(bodyInit, undefined, bodyInit.byteLength);
                const readable = Readable.from(buffer);
                const body = new PonyfillReadableStream(readable);
                return body;
            },
        };
    }
    if (bodyInit instanceof Readable) {
        return {
            bodyType: BodyInitType.Readable,
            contentType: null,
            contentLength: null,
            bodyFactory() {
                const body = new PonyfillReadableStream(bodyInit);
                return body;
            },
        };
    }
    if ('stream' in bodyInit) {
        return {
            contentType: bodyInit.type,
            contentLength: bodyInit.size,
            bodyFactory() {
                const bodyStream = bodyInit.stream();
                const body = new PonyfillReadableStream(bodyStream);
                return body;
            },
        };
    }
    if (bodyInit instanceof URLSearchParams) {
        const contentType = 'application/x-www-form-urlencoded;charset=UTF-8';
        return {
            bodyType: BodyInitType.String,
            contentType,
            contentLength: null,
            bodyFactory() {
                const body = new PonyfillReadableStream(Readable.from(bodyInit.toString()));
                return body;
            },
        };
    }
    if ('forEach' in bodyInit) {
        const boundary = Math.random().toString(36).substr(2);
        const contentType = `multipart/form-data; boundary=${boundary}`;
        return {
            contentType,
            contentLength: null,
            bodyFactory() {
                return getStreamFromFormData(bodyInit, boundary);
            },
        };
    }
    if (bodyInit[Symbol.iterator] || bodyInit[Symbol.asyncIterator]) {
        return {
            contentType: null,
            contentLength: null,
            bodyFactory() {
                const readable = Readable.from(bodyInit);
                return new PonyfillReadableStream(readable);
            },
        };
    }
    throw new Error('Unknown body type');
}

class PonyfillHeaders {
    constructor(headersInit) {
        this.map = new Map();
        if (headersInit != null) {
            if (Array.isArray(headersInit)) {
                for (const [key, value] of headersInit) {
                    if (Array.isArray(value)) {
                        for (const v of value) {
                            this.append(key, v);
                        }
                    }
                    else {
                        this.map.set(key, value);
                    }
                }
            }
            else if ('get' in headersInit) {
                headersInit.forEach((value, key) => {
                    this.append(key, value);
                });
            }
            else {
                for (const key in headersInit) {
                    const value = headersInit[key];
                    if (Array.isArray(value)) {
                        for (const v of value) {
                            this.append(key, v);
                        }
                    }
                    else if (value != null) {
                        this.set(key, value);
                    }
                }
            }
        }
    }
    append(name, value) {
        const key = name.toLowerCase();
        if (this.map.has(key)) {
            value = this.map.get(key) + ', ' + value;
        }
        this.map.set(key, value);
    }
    get(name) {
        const key = name.toLowerCase();
        return this.map.get(key) || null;
    }
    has(name) {
        const key = name.toLowerCase();
        return this.map.has(key);
    }
    set(name, value) {
        const key = name.toLowerCase();
        this.map.set(key, value);
    }
    delete(name) {
        const key = name.toLowerCase();
        this.map.delete(key);
    }
    forEach(callback) {
        this.map.forEach((value, key) => {
            callback(value, key, this);
        });
    }
    entries() {
        return this.map.entries();
    }
    [Symbol.iterator]() {
        return this.map.entries();
    }
}

function isRequest(input) {
    return input[Symbol.toStringTag] === 'Request';
}
class PonyfillRequest extends PonyfillBody {
    constructor(input, options) {
        var _a;
        let url;
        let bodyInit = null;
        let requestInit;
        if (typeof input === 'string') {
            url = input;
        }
        else if (input instanceof URL) {
            url = input.toString();
        }
        else if (isRequest(input)) {
            url = input.url;
            bodyInit = input.body;
            requestInit = input;
        }
        if (options != null) {
            bodyInit = options.body || null;
            requestInit = options;
        }
        super(bodyInit, options);
        this.destination = '';
        this.priority = 'auto';
        this.cache = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.cache) || 'default';
        this.credentials = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.credentials) || 'same-origin';
        this.headers = new PonyfillHeaders(requestInit === null || requestInit === void 0 ? void 0 : requestInit.headers);
        this.integrity = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.integrity) || '';
        this.keepalive = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.keepalive) != null ? requestInit === null || requestInit === void 0 ? void 0 : requestInit.keepalive : false;
        this.method = ((_a = requestInit === null || requestInit === void 0 ? void 0 : requestInit.method) === null || _a === void 0 ? void 0 : _a.toUpperCase()) || 'GET';
        this.mode = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.mode) || 'cors';
        this.redirect = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.redirect) || 'follow';
        this.referrer = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.referrer) || 'about:client';
        this.referrerPolicy = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.referrerPolicy) || 'no-referrer';
        this.signal = (requestInit === null || requestInit === void 0 ? void 0 : requestInit.signal) || new PonyfillAbortController().signal;
        this.url = url || '';
        const contentTypeInHeaders = this.headers.get('content-type');
        if (!contentTypeInHeaders) {
            if (this.contentType) {
                this.headers.set('content-type', this.contentType);
            }
        }
        else {
            this.contentType = contentTypeInHeaders;
        }
        const contentLengthInHeaders = this.headers.get('content-length');
        if (!contentLengthInHeaders) {
            if (this.contentLength) {
                this.headers.set('content-length', this.contentLength.toString());
            }
        }
        else {
            this.contentLength = parseInt(contentLengthInHeaders, 10);
        }
    }
    clone() {
        return new Request(this);
    }
}

class PonyfillResponse extends PonyfillBody {
    constructor(body, init) {
        super(body || null, init);
        this.headers = new PonyfillHeaders();
        this.status = 200;
        this.statusText = 'OK';
        this.url = '';
        this.redirected = false;
        this.type = 'default';
        if (init) {
            this.headers = new PonyfillHeaders(init.headers);
            this.status = init.status || 200;
            this.statusText = init.statusText || 'OK';
            this.url = init.url || '';
            this.redirected = init.redirected || false;
            this.type = init.type || 'default';
        }
        const contentTypeInHeaders = this.headers.get('content-type');
        if (!contentTypeInHeaders) {
            if (this.contentType) {
                this.headers.set('content-type', this.contentType);
            }
        }
        else {
            this.contentType = contentTypeInHeaders;
        }
        const contentLengthInHeaders = this.headers.get('content-length');
        if (!contentLengthInHeaders) {
            if (this.contentLength) {
                this.headers.set('content-length', this.contentLength.toString());
            }
        }
        else {
            this.contentLength = parseInt(contentLengthInHeaders, 10);
        }
    }
    get ok() {
        return this.status >= 200 && this.status < 300;
    }
    clone() {
        return new PonyfillResponse(this.body, this);
    }
    static error() {
        return new PonyfillResponse(null, {
            status: 500,
            statusText: 'Internal Server Error',
        });
    }
    static redirect(url, status = 301) {
        if (status < 300 || status > 399) {
            throw new RangeError('Invalid status code');
        }
        return new PonyfillResponse(null, {
            headers: {
                location: url,
            },
            status,
        });
    }
    static json(data, init = {}) {
        return new PonyfillResponse(JSON.stringify(data), {
            ...init,
            headers: {
                'Content-Type': 'application/json',
                ...init === null || init === void 0 ? void 0 : init.headers,
            },
        });
    }
}

function getHeadersObj(headers) {
    if (headers == null || !('forEach' in headers)) {
        return headers;
    }
    const obj = {};
    headers.forEach((value, key) => {
        obj[key] = value;
    });
    return obj;
}

function getResponseForFile(url) {
    const path = fileURLToPath(url);
    const readable = createReadStream(path);
    return new PonyfillResponse(readable);
}
function getRequestFnForProtocol(protocol) {
    switch (protocol) {
        case 'http:':
            return request$1;
        case 'https:':
            return request;
    }
    throw new Error(`Unsupported protocol: ${protocol}`);
}
const BASE64_SUFFIX = ';base64';
function fetchPonyfill(info, init) {
    if (typeof info === 'string' || info instanceof URL) {
        const ponyfillRequest = new PonyfillRequest(info, init);
        return fetchPonyfill(ponyfillRequest);
    }
    const fetchRequest = info;
    return new Promise((resolve, reject) => {
        try {
            const url = new URL(fetchRequest.url, 'http://localhost');
            if (url.protocol === 'data:') {
                const [mimeType = 'text/plain', ...datas] = url.pathname.split(',');
                const data = decodeURIComponent(datas.join(','));
                if (mimeType.endsWith(BASE64_SUFFIX)) {
                    const buffer = Buffer.from(data, 'base64');
                    const realMimeType = mimeType.slice(0, -BASE64_SUFFIX.length);
                    const file = new PonyfillBlob([buffer], { type: realMimeType });
                    const response = new PonyfillResponse(file, {
                        status: 200,
                        statusText: 'OK',
                    });
                    resolve(response);
                    return;
                }
                const response = new PonyfillResponse(data, {
                    status: 200,
                    statusText: 'OK',
                    headers: {
                        'content-type': mimeType,
                    },
                });
                resolve(response);
                return;
            }
            if (url.protocol === 'file:') {
                const response = getResponseForFile(url);
                resolve(response);
                return;
            }
            const requestFn = getRequestFnForProtocol(url.protocol);
            const nodeReadable = (fetchRequest.body != null
                ? 'pipe' in fetchRequest.body
                    ? fetchRequest.body
                    : Readable.from(fetchRequest.body)
                : null);
            const nodeHeaders = getHeadersObj(fetchRequest.headers);
            const abortListener = function abortListener(event) {
                nodeRequest.destroy();
                const reason = event.detail;
                reject(new PonyfillAbortError(reason));
            };
            fetchRequest.signal.addEventListener('abort', abortListener);
            const nodeRequest = requestFn(url, {
                // signal: fetchRequest.signal will be added when v14 reaches EOL
                method: fetchRequest.method,
                headers: nodeHeaders,
            });
            nodeRequest.once('response', nodeResponse => {
                if (nodeResponse.headers.location) {
                    if (fetchRequest.redirect === 'error') {
                        const redirectError = new Error('Redirects are not allowed');
                        reject(redirectError);
                        nodeResponse.resume();
                        return;
                    }
                    if (fetchRequest.redirect === 'follow') {
                        const redirectedUrl = new URL(nodeResponse.headers.location, url);
                        const redirectResponse$ = fetchPonyfill(redirectedUrl, info);
                        resolve(redirectResponse$.then(redirectResponse => {
                            redirectResponse.redirected = true;
                            return redirectResponse;
                        }));
                        nodeResponse.resume();
                        return;
                    }
                }
                const responseHeaders = nodeResponse.headers;
                const ponyfillResponse = new PonyfillResponse(nodeResponse, {
                    status: nodeResponse.statusCode,
                    statusText: nodeResponse.statusMessage,
                    headers: responseHeaders,
                    url: info.url,
                });
                resolve(ponyfillResponse);
            });
            nodeRequest.once('error', reject);
            if (nodeReadable) {
                nodeReadable.pipe(nodeRequest);
            }
            else {
                nodeRequest.end();
            }
        }
        catch (e) {
            reject(e);
        }
    });
}

class PonyfillTextEncoder {
    constructor(encoding = 'utf-8') {
        this.encoding = encoding;
    }
    encode(input) {
        return Buffer.from(input, this.encoding);
    }
    encodeInto(source, destination) {
        const buffer = this.encode(source);
        const copied = buffer.copy(destination);
        return {
            read: copied,
            written: copied,
        };
    }
}
class PonyfillTextDecoder {
    constructor(encoding = 'utf-8', options) {
        this.encoding = encoding;
        this.fatal = false;
        this.ignoreBOM = false;
        if (options) {
            this.fatal = options.fatal || false;
            this.ignoreBOM = options.ignoreBOM || false;
        }
    }
    decode(input) {
        return Buffer.from(input).toString(this.encoding);
    }
}
function PonyfillBtoa(input) {
    return Buffer.from(input, 'binary').toString('base64');
}

export { PonyfillAbortController as AbortController, PonyfillAbortError as AbortError, PonyfillAbortSignal as AbortSignal, PonyfillBlob as Blob, PonyfillBody as Body, PonyfillFile as File, PonyfillFormData as FormData, PonyfillHeaders as Headers, PonyfillReadableStream as ReadableStream, PonyfillRequest as Request, PonyfillResponse as Response, PonyfillTextDecoder as TextDecoder, PonyfillTextEncoder as TextEncoder, PonyfillBtoa as btoa, fetchPonyfill as fetch };
