'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var shopifyApi = require('@shopify/shopify-api');

// import type {PrismaClient} from '@prisma/client';
class PrismaSessionStorage {
  constructor(prisma) {
    this.prisma = prisma;
    this.ready = void 0;
    if (this.prisma.session === undefined) {
      throw new Error('PrismaClient does not have a Session table');
    }
    this.ready = this.prisma.session.count().catch(() => {
      throw new MissingSessionTableError('Prisma Session table does not exist. This could happen for a few reasons, see https://github.com/Shopify/shopify-app-js/tree/main/packages/shopify-app-session-storage-prisma#troubleshooting for more information');
    });
  }
  async storeSession(session) {
    await this.ready;
    const data = this.sessionToRow(session);
    await this.prisma.session.upsert({
      where: {
        id: session.id
      },
      update: data,
      create: data
    });
    return true;
  }
  async loadSession(id) {
    await this.ready;
    const row = await this.prisma.session.findUnique({
      where: {
        id
      }
    });
    if (!row) {
      return undefined;
    }
    return this.rowToSession(row);
  }
  async deleteSession(id) {
    await this.ready;
    try {
      await this.prisma.session.delete({
        where: {
          id
        }
      });
    } catch {
      return true;
    }
    return true;
  }
  async deleteSessions(ids) {
    await this.ready;
    await this.prisma.session.deleteMany({
      where: {
        id: {
          in: ids
        }
      }
    });
    return true;
  }
  async findSessionsByShop(shop) {
    await this.ready;
    const sessions = await this.prisma.session.findMany({
      where: {
        shop
      },
      take: 25,
      orderBy: [{
        expires: 'desc'
      }]
    });
    return sessions.map(session => this.rowToSession(session));
  }
  sessionToRow(session) {
    var _sessionParams$online;
    const sessionParams = session.toObject();
    return {
      id: session.id,
      shop: session.shop,
      state: session.state,
      isOnline: session.isOnline,
      scope: session.scope || null,
      expires: session.expires || null,
      accessToken: session.accessToken || '',
      userId: ((_sessionParams$online = sessionParams.onlineAccessInfo) === null || _sessionParams$online === void 0 ? void 0 : _sessionParams$online.associated_user.id) || null
    };
  }
  rowToSession(row) {
    const sessionParams = {
      id: row.id,
      shop: row.shop,
      state: row.state,
      isOnline: row.isOnline
    };
    if (row.expires) {
      sessionParams.expires = row.expires.getTime();
    }
    if (row.scope) {
      sessionParams.scope = row.scope;
    }
    if (row.accessToken) {
      sessionParams.accessToken = row.accessToken;
    }
    if (row.userId) {
      sessionParams.onlineAccessInfo = String(row.userId);
    }
    return shopifyApi.Session.fromPropertyArray(Object.entries(sessionParams));
  }
}
class MissingSessionTableError extends Error {}

exports.MissingSessionTableError = MissingSessionTableError;
exports.PrismaSessionStorage = PrismaSessionStorage;
