"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthStrategy = void 0;
var tslib_1 = require("tslib");
var server_runtime_1 = require("@remix-run/server-runtime");
var shopify_api_1 = require("@shopify/shopify-api");
var billing_1 = require("../../billing");
var helpers_1 = require("../helpers");
var graphql_client_1 = require("./graphql-client");
var rest_client_1 = require("./rest-client");
var SESSION_TOKEN_PARAM = 'id_token';
var AuthStrategy = /** @class */ (function () {
    function AuthStrategy(_a) {
        var api = _a.api, config = _a.config, logger = _a.logger;
        this.api = api;
        this.config = config;
        this.logger = logger;
    }
    AuthStrategy.prototype.authenticateAdmin = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, logger, config, cors, sessionContext, errorOrResponse_1, context;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this, api = _a.api, logger = _a.logger, config = _a.config;
                        (0, helpers_1.rejectBotRequest)({ api: api, logger: logger, config: config }, request);
                        (0, helpers_1.respondToOptionsRequest)({ api: api, logger: logger, config: config }, request);
                        cors = (0, helpers_1.ensureCORSHeadersFactory)({ api: api, logger: logger, config: config }, request);
                        _b.label = 1;
                    case 1:
                        _b.trys.push([1, 3, , 4]);
                        return [4 /*yield*/, this.authenticateAndGetSessionContext(request)];
                    case 2:
                        sessionContext = _b.sent();
                        return [3 /*break*/, 4];
                    case 3:
                        errorOrResponse_1 = _b.sent();
                        if (errorOrResponse_1 instanceof Response) {
                            cors(errorOrResponse_1);
                        }
                        throw errorOrResponse_1;
                    case 4:
                        context = {
                            admin: this.createAdminApiContext(request, sessionContext.session),
                            billing: this.createBillingContext(request, sessionContext.session),
                            session: sessionContext.session,
                            cors: cors,
                        };
                        if (config.isEmbeddedApp) {
                            return [2 /*return*/, tslib_1.__assign(tslib_1.__assign({}, context), { sessionToken: sessionContext.token })];
                        }
                        else {
                            return [2 /*return*/, context];
                        }
                        return [2 /*return*/];
                }
            });
        });
    };
    AuthStrategy.prototype.authenticateAndGetSessionContext = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, logger, config, url, isPatchSessionToken, isExitIframe, isAuthRequest, isAuthCallbackRequest, sessionTokenHeader, destination, sessionToken;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this, api = _a.api, logger = _a.logger, config = _a.config;
                        url = new URL(request.url);
                        isPatchSessionToken = url.pathname === config.auth.patchSessionTokenPath;
                        isExitIframe = url.pathname === config.auth.exitIframePath;
                        isAuthRequest = url.pathname === config.auth.path;
                        isAuthCallbackRequest = url.pathname === config.auth.callbackPath;
                        sessionTokenHeader = (0, helpers_1.getSessionTokenHeader)(request);
                        logger.info('Authenticating admin request');
                        if (!isPatchSessionToken) return [3 /*break*/, 1];
                        logger.debug('Rendering bounce page');
                        throw this.renderAppBridge(request);
                    case 1:
                        if (!isExitIframe) return [3 /*break*/, 2];
                        destination = url.searchParams.get('exitIframe');
                        logger.debug('Rendering exit iframe page', { destination: destination });
                        throw this.renderAppBridge(request, destination);
                    case 2:
                        if (!isAuthCallbackRequest) return [3 /*break*/, 4];
                        return [4 /*yield*/, this.handleAuthCallbackRequest(request)];
                    case 3: throw _b.sent();
                    case 4:
                        if (!isAuthRequest) return [3 /*break*/, 6];
                        return [4 /*yield*/, this.handleAuthBeginRequest(request)];
                    case 5: throw _b.sent();
                    case 6:
                        if (!sessionTokenHeader) return [3 /*break*/, 8];
                        return [4 /*yield*/, (0, helpers_1.validateSessionToken)({ api: api, logger: logger, config: config }, sessionTokenHeader)];
                    case 7:
                        sessionToken = _b.sent();
                        return [2 /*return*/, this.validateAuthenticatedSession(request, sessionToken)];
                    case 8: return [4 /*yield*/, this.validateUrlParams(request)];
                    case 9:
                        _b.sent();
                        return [4 /*yield*/, this.ensureInstalledOnShop(request)];
                    case 10:
                        _b.sent();
                        return [4 /*yield*/, this.ensureAppIsEmbeddedIfRequired(request)];
                    case 11:
                        _b.sent();
                        return [4 /*yield*/, this.ensureSessionTokenSearchParamIfRequired(request)];
                    case 12:
                        _b.sent();
                        return [2 /*return*/, this.ensureSessionExists(request)];
                }
            });
        });
    };
    AuthStrategy.prototype.handleAuthBeginRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, config, logger, shop;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this, api = _a.api, config = _a.config, logger = _a.logger;
                        logger.info('Handling OAuth begin request');
                        shop = this.ensureValidShopParam(request);
                        logger.debug('OAuth request contained valid shop', { shop: shop });
                        if (!(config.isEmbeddedApp &&
                            request.headers.get('Sec-Fetch-Dest') === 'iframe')) return [3 /*break*/, 1];
                        logger.debug('Auth request in iframe detected, exiting iframe', { shop: shop });
                        throw (0, helpers_1.redirectWithExitIframe)({ api: api, config: config, logger: logger }, request, shop);
                    case 1: return [4 /*yield*/, (0, helpers_1.beginAuth)({ api: api, config: config, logger: logger }, request, false, shop)];
                    case 2: throw _b.sent();
                }
            });
        });
    };
    AuthStrategy.prototype.handleAuthCallbackRequest = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, config, logger, shop, _b, session, responseHeaders, error_1;
            return tslib_1.__generator(this, function (_c) {
                switch (_c.label) {
                    case 0:
                        _a = this, api = _a.api, config = _a.config, logger = _a.logger;
                        logger.info('Handling OAuth callback request');
                        shop = this.ensureValidShopParam(request);
                        _c.label = 1;
                    case 1:
                        _c.trys.push([1, 9, , 13]);
                        return [4 /*yield*/, api.auth.callback({
                                rawRequest: request,
                            })];
                    case 2:
                        _b = _c.sent(), session = _b.session, responseHeaders = _b.headers;
                        return [4 /*yield*/, config.sessionStorage.storeSession(session)];
                    case 3:
                        _c.sent();
                        if (!(config.useOnlineTokens && !session.isOnline)) return [3 /*break*/, 5];
                        logger.info('Requesting online access token for offline session');
                        return [4 /*yield*/, (0, helpers_1.beginAuth)({ api: api, config: config, logger: logger }, request, true, shop)];
                    case 4:
                        _c.sent();
                        _c.label = 5;
                    case 5:
                        if (!config.hooks.afterAuth) return [3 /*break*/, 7];
                        logger.info('Running afterAuth hook');
                        return [4 /*yield*/, config.hooks.afterAuth({
                                session: session,
                                admin: this.createAdminApiContext(request, session),
                            })];
                    case 6:
                        _c.sent();
                        _c.label = 7;
                    case 7: return [4 /*yield*/, this.redirectToShopifyOrAppRoot(request, responseHeaders)];
                    case 8: throw _c.sent();
                    case 9:
                        error_1 = _c.sent();
                        if (error_1 instanceof Response) {
                            throw error_1;
                        }
                        logger.error('Error during OAuth callback', { error: error_1.message });
                        if (!(error_1 instanceof shopify_api_1.CookieNotFound)) return [3 /*break*/, 11];
                        return [4 /*yield*/, this.handleAuthBeginRequest(request)];
                    case 10: throw _c.sent();
                    case 11:
                        if (error_1 instanceof shopify_api_1.InvalidHmacError ||
                            error_1 instanceof shopify_api_1.InvalidOAuthError) {
                            throw new Response(undefined, {
                                status: 400,
                                statusText: 'Invalid OAuth Request',
                            });
                        }
                        else {
                            throw new Response(undefined, {
                                status: 500,
                                statusText: 'Internal Server Error',
                            });
                        }
                        _c.label = 12;
                    case 12: return [3 /*break*/, 13];
                    case 13: return [2 /*return*/];
                }
            });
        });
    };
    AuthStrategy.prototype.validateUrlParams = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, config, logger, url, shop, host;
            return tslib_1.__generator(this, function (_b) {
                _a = this, api = _a.api, config = _a.config, logger = _a.logger;
                if (config.isEmbeddedApp) {
                    url = new URL(request.url);
                    shop = api.utils.sanitizeShop(url.searchParams.get('shop'));
                    if (!shop) {
                        logger.debug('Missing or invalid shop, redirecting to login path', {
                            shop: shop,
                        });
                        throw (0, server_runtime_1.redirect)(config.auth.loginPath);
                    }
                    host = api.utils.sanitizeHost(url.searchParams.get('host'));
                    if (!host) {
                        logger.debug('Invalid host, redirecting to login path', {
                            host: url.searchParams.get('host'),
                        });
                        throw (0, server_runtime_1.redirect)(config.auth.loginPath);
                    }
                }
                return [2 /*return*/];
            });
        });
    };
    AuthStrategy.prototype.ensureInstalledOnShop = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, config, logger, url, shop, isEmbedded, offlineId, _b, offlineSession, error_2, message, context;
            return tslib_1.__generator(this, function (_c) {
                switch (_c.label) {
                    case 0:
                        _a = this, api = _a.api, config = _a.config, logger = _a.logger;
                        url = new URL(request.url);
                        shop = url.searchParams.get('shop');
                        isEmbedded = url.searchParams.get('embedded') === '1';
                        // Ensure app is installed
                        logger.debug('Ensuring app is installed on shop', { shop: shop });
                        if (!shop) return [3 /*break*/, 1];
                        _b = api.session.getOfflineId(shop);
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, api.session.getCurrentId({ isOnline: false, rawRequest: request })];
                    case 2:
                        _b = _c.sent();
                        _c.label = 3;
                    case 3:
                        offlineId = _b;
                        if (!offlineId) {
                            logger.info("Could not find a shop, can't authenticate request");
                            throw new Response(undefined, {
                                status: 400,
                                statusText: 'Bad Request',
                            });
                        }
                        return [4 /*yield*/, config.sessionStorage.loadSession(offlineId)];
                    case 4:
                        offlineSession = _c.sent();
                        if (!!offlineSession) return [3 /*break*/, 7];
                        logger.info("Shop hasn't installed app yet, redirecting to OAuth", {
                            shop: shop,
                        });
                        if (!isEmbedded) return [3 /*break*/, 5];
                        (0, helpers_1.redirectWithExitIframe)({ api: api, config: config, logger: logger }, request, shop);
                        return [3 /*break*/, 7];
                    case 5: return [4 /*yield*/, (0, helpers_1.beginAuth)({ api: api, config: config, logger: logger }, request, false, shop)];
                    case 6: throw _c.sent();
                    case 7:
                        shop = shop || offlineSession.shop;
                        if (!(config.isEmbeddedApp && !isEmbedded)) return [3 /*break*/, 16];
                        _c.label = 8;
                    case 8:
                        _c.trys.push([8, 10, , 16]);
                        logger.debug('Ensuring offline session is valid before embedding', {
                            shop: shop,
                        });
                        return [4 /*yield*/, this.testSession(offlineSession)];
                    case 9:
                        _c.sent();
                        logger.debug('Offline session is still valid, embedding app', { shop: shop });
                        return [3 /*break*/, 16];
                    case 10:
                        error_2 = _c.sent();
                        if (!(error_2 instanceof shopify_api_1.HttpResponseError)) return [3 /*break*/, 14];
                        if (!(error_2.response.code === 401)) return [3 /*break*/, 12];
                        logger.info('Shop session is no longer valid, redirecting to OAuth', { shop: shop });
                        return [4 /*yield*/, (0, helpers_1.beginAuth)({ api: api, config: config, logger: logger }, request, false, shop)];
                    case 11: throw _c.sent();
                    case 12:
                        message = JSON.stringify(error_2.response.body, null, 2);
                        logger.error("Unexpected error during session validation: ".concat(message), { shop: shop });
                        throw new Response(undefined, {
                            status: error_2.response.code,
                            statusText: error_2.response.statusText,
                        });
                    case 13: return [3 /*break*/, 15];
                    case 14:
                        if (error_2 instanceof shopify_api_1.GraphqlQueryError) {
                            context = { shop: shop };
                            if (error_2.response) {
                                context.response = JSON.stringify(error_2.response);
                            }
                            logger.error("Unexpected error during session validation: ".concat(error_2.message), context);
                            throw new Response(undefined, {
                                status: 500,
                                statusText: 'Internal Server Error',
                            });
                        }
                        _c.label = 15;
                    case 15: return [3 /*break*/, 16];
                    case 16: return [2 /*return*/];
                }
            });
        });
    };
    AuthStrategy.prototype.testSession = function (session) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var api, client;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        api = this.api;
                        client = new api.clients.Graphql({
                            session: session,
                        });
                        return [4 /*yield*/, client.query({
                                data: "#graphql\n        query shopifyAppShopName {\n          shop {\n            name\n          }\n        }\n      ",
                            })];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    AuthStrategy.prototype.ensureValidShopParam = function (request) {
        var url = new URL(request.url);
        var api = this.api;
        var shop = api.utils.sanitizeShop(url.searchParams.get('shop'));
        if (!shop) {
            throw new Response('Shop param is invalid', {
                status: 400,
            });
        }
        return shop;
    };
    AuthStrategy.prototype.ensureAppIsEmbeddedIfRequired = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, logger, url, shop;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this, api = _a.api, logger = _a.logger;
                        url = new URL(request.url);
                        shop = url.searchParams.get('shop');
                        if (!(api.config.isEmbeddedApp && url.searchParams.get('embedded') !== '1')) return [3 /*break*/, 2];
                        logger.debug('App is not embedded, redirecting to Shopify', { shop: shop });
                        return [4 /*yield*/, this.redirectToShopifyOrAppRoot(request)];
                    case 1:
                        _b.sent();
                        _b.label = 2;
                    case 2: return [2 /*return*/];
                }
            });
        });
    };
    AuthStrategy.prototype.ensureSessionTokenSearchParamIfRequired = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, logger, url, shop, searchParamSessionToken;
            return tslib_1.__generator(this, function (_b) {
                _a = this, api = _a.api, logger = _a.logger;
                url = new URL(request.url);
                shop = url.searchParams.get('shop');
                searchParamSessionToken = url.searchParams.get(SESSION_TOKEN_PARAM);
                if (api.config.isEmbeddedApp && !searchParamSessionToken) {
                    logger.debug('Missing session token in search params, going to bounce page', { shop: shop });
                    this.redirectToBouncePage(url);
                }
                return [2 /*return*/];
            });
        });
    };
    AuthStrategy.prototype.ensureSessionExists = function (request) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, config, logger, url, shop, searchParamSessionToken, sessionToken, sessionId;
            var _b;
            return tslib_1.__generator(this, function (_c) {
                switch (_c.label) {
                    case 0:
                        _a = this, api = _a.api, config = _a.config, logger = _a.logger;
                        url = new URL(request.url);
                        shop = url.searchParams.get('shop');
                        searchParamSessionToken = url.searchParams.get(SESSION_TOKEN_PARAM);
                        if (!api.config.isEmbeddedApp) return [3 /*break*/, 2];
                        logger.debug('Session token is present in query params, validating session', { shop: shop });
                        return [4 /*yield*/, (0, helpers_1.validateSessionToken)({ api: api, config: config, logger: logger }, searchParamSessionToken)];
                    case 1:
                        sessionToken = _c.sent();
                        return [2 /*return*/, this.validateAuthenticatedSession(request, sessionToken)];
                    case 2: return [4 /*yield*/, api.session.getCurrentId({
                            isOnline: config.useOnlineTokens,
                            rawRequest: request,
                        })];
                    case 3:
                        sessionId = _c.sent();
                        if (!!sessionId) return [3 /*break*/, 5];
                        logger.debug('Session id not found in cookies, redirecting to OAuth', {
                            shop: shop,
                        });
                        return [4 /*yield*/, (0, helpers_1.beginAuth)({ api: api, config: config, logger: logger }, request, false, shop)];
                    case 4: throw _c.sent();
                    case 5:
                        _b = {};
                        return [4 /*yield*/, this.loadSession(request, shop, sessionId)];
                    case 6: return [2 /*return*/, (_b.session = _c.sent(), _b)];
                }
            });
        });
    };
    AuthStrategy.prototype.validateAuthenticatedSession = function (request, payload) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, config, logger, api, dest, shop, sessionId, session;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this, config = _a.config, logger = _a.logger, api = _a.api;
                        dest = new URL(payload.dest);
                        shop = dest.hostname;
                        sessionId = config.useOnlineTokens
                            ? api.session.getJwtSessionId(shop, payload.sub)
                            : api.session.getOfflineId(shop);
                        return [4 /*yield*/, this.loadSession(request, shop, sessionId)];
                    case 1:
                        session = _b.sent();
                        logger.debug('Found session, request is valid', { shop: shop });
                        return [2 /*return*/, { session: session, token: payload }];
                }
            });
        });
    };
    AuthStrategy.prototype.loadSession = function (request, shop, sessionId) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, api, config, logger, session;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this, api = _a.api, config = _a.config, logger = _a.logger;
                        logger.debug('Loading session from storage', { sessionId: sessionId });
                        return [4 /*yield*/, config.sessionStorage.loadSession(sessionId)];
                    case 1:
                        session = _b.sent();
                        if (!!session) return [3 /*break*/, 3];
                        logger.debug('No session found, redirecting to OAuth', { shop: shop });
                        return [4 /*yield*/, (0, helpers_1.redirectToAuthPage)({ api: api, config: config, logger: logger }, request, shop)];
                    case 2:
                        _b.sent();
                        return [3 /*break*/, 5];
                    case 3:
                        if (!!session.isActive(config.scopes)) return [3 /*break*/, 5];
                        logger.debug('Found a session, but it has expired, redirecting to OAuth', { shop: shop });
                        return [4 /*yield*/, (0, helpers_1.redirectToAuthPage)({ api: api, config: config, logger: logger }, request, shop)];
                    case 4:
                        _b.sent();
                        _b.label = 5;
                    case 5: return [2 /*return*/, session];
                }
            });
        });
    };
    AuthStrategy.prototype.redirectToShopifyOrAppRoot = function (request, responseHeaders) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var api, url, host, shop, redirectUrl, _a;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        api = this.api;
                        url = new URL(request.url);
                        host = api.utils.sanitizeHost(url.searchParams.get('host'));
                        shop = api.utils.sanitizeShop(url.searchParams.get('shop'));
                        if (!api.config.isEmbeddedApp) return [3 /*break*/, 2];
                        return [4 /*yield*/, api.auth.getEmbeddedAppUrl({ rawRequest: request })];
                    case 1:
                        _a = _b.sent();
                        return [3 /*break*/, 3];
                    case 2:
                        _a = "/?shop=".concat(shop, "&host=").concat(encodeURIComponent(host));
                        _b.label = 3;
                    case 3:
                        redirectUrl = _a;
                        throw (0, server_runtime_1.redirect)(redirectUrl, { headers: responseHeaders });
                }
            });
        });
    };
    AuthStrategy.prototype.redirectToBouncePage = function (url) {
        var _a = this, api = _a.api, config = _a.config;
        // eslint-disable-next-line no-warning-comments
        // TODO this is to work around a remix bug
        // https://github.com/orgs/Shopify/projects/6899/views/1?pane=issue&itemId=28376650
        url.protocol = "".concat(api.config.hostScheme, ":");
        var params = new URLSearchParams(url.search);
        params.set('shopify-reload', url.href);
        // eslint-disable-next-line no-warning-comments
        // TODO Make sure this works on chrome without a tunnel (weird HTTPS redirect issue)
        // https://github.com/orgs/Shopify/projects/6899/views/1?pane=issue&itemId=28376650
        throw (0, server_runtime_1.redirect)("".concat(config.auth.patchSessionTokenPath, "?").concat(params.toString()));
    };
    AuthStrategy.prototype.renderAppBridge = function (request, redirectTo) {
        var config = this.config;
        var redirectToScript = '';
        if (redirectTo) {
            var redirectUrl = decodeURIComponent(redirectTo.startsWith('/')
                ? "".concat(config.appUrl).concat(redirectTo)
                : redirectTo);
            redirectToScript = "<script>window.open(\"".concat(redirectUrl, "\", \"_top\")</script>");
        }
        var responseHeaders = new Headers({
            'content-type': 'text/html;charset=utf-8',
        });
        (0, helpers_1.addDocumentResponseHeaders)(responseHeaders, config.isEmbeddedApp, new URL(request.url).searchParams.get('shop'));
        throw new Response("\n        <script data-api-key=\"".concat(config.apiKey, "\" src=\"").concat((0, helpers_1.appBridgeUrl)(), "\"></script>\n        ").concat(redirectToScript, "\n      "), { headers: responseHeaders });
    };
    AuthStrategy.prototype.overriddenRestClient = function (request, session) {
        var _a = this, api = _a.api, config = _a.config, logger = _a.logger;
        var client = new rest_client_1.RemixRestClient({
            params: { api: api, config: config, logger: logger },
            request: request,
            session: session,
        });
        if (api.rest) {
            client.resources = {};
            var RestResourceClient_1 = (0, rest_client_1.restResourceClientFactory)({
                params: { api: api, config: config, logger: logger },
                request: request,
                session: session,
            });
            Object.entries(api.rest).forEach(function (_a) {
                var _b = tslib_1.__read(_a, 2), name = _b[0], resource = _b[1];
                var RemixResource = /** @class */ (function (_super) {
                    tslib_1.__extends(RemixResource, _super);
                    function RemixResource() {
                        return _super !== null && _super.apply(this, arguments) || this;
                    }
                    RemixResource.Client = RestResourceClient_1;
                    return RemixResource;
                }(resource));
                Reflect.defineProperty(RemixResource, 'name', {
                    value: name,
                });
                Reflect.set(client.resources, name, RemixResource);
            });
        }
        return client;
    };
    AuthStrategy.prototype.createBillingContext = function (request, session) {
        var _a = this, api = _a.api, logger = _a.logger, config = _a.config;
        return {
            require: (0, billing_1.requireBillingFactory)({ api: api, logger: logger, config: config }, request, session),
            request: (0, billing_1.requestBillingFactory)({ api: api, logger: logger, config: config }, request, session),
            cancel: (0, billing_1.cancelBillingFactory)({ api: api, logger: logger, config: config }, request, session),
        };
    };
    AuthStrategy.prototype.createAdminApiContext = function (request, session) {
        var _a = this, api = _a.api, config = _a.config, logger = _a.logger;
        return {
            rest: this.overriddenRestClient(request, session),
            graphql: (0, graphql_client_1.graphqlClientFactory)({
                params: { api: api, config: config, logger: logger },
                request: request,
                session: session,
            }),
        };
    };
    return AuthStrategy;
}());
exports.AuthStrategy = AuthStrategy;
//# sourceMappingURL=authenticate.js.map