'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var shopifyApi = require('@shopify/shopify-api');
var serverRuntime = require('@remix-run/server-runtime');
var redirectToAuthPage = require('../auth/helpers/redirect-to-auth-page.js');
require('isbot');
var redirectWithAppBridgeHeaders = require('../auth/helpers/redirect-with-app-bridge-headers.js');

function requestBillingFactory(params, request, session) {
  return async function requestBilling({
    plan,
    isTest,
    returnUrl
  }) {
    const {
      api,
      logger
    } = params;
    logger.info('Requesting billing', {
      shop: session.shop,
      plan,
      isTest,
      returnUrl
    });
    let result;
    try {
      result = await api.billing.request({
        plan: plan,
        session,
        isTest,
        returnUrl,
        returnObject: true
      });
    } catch (error) {
      if (error instanceof shopifyApi.HttpResponseError && error.response.code === 401) {
        logger.debug('API token was invalid, redirecting to OAuth', {
          shop: session.shop
        });
        throw await redirectToAuthPage.redirectToAuthPage(params, request, session.shop);
      } else {
        throw error;
      }
    }
    throw redirectOutOfApp(params, request, result.confirmationUrl, session.shop);
  };
}
function redirectOutOfApp(params, request, url, shop) {
  const {
    config,
    logger
  } = params;
  logger.debug('Redirecting out of app', {
    url
  });
  const requestUrl = new URL(request.url);
  const isEmbeddedRequest = requestUrl.searchParams.get('embedded') === '1';
  const isXhrRequest = request.headers.get('authorization');
  if (isXhrRequest) {
    // eslint-disable-next-line no-warning-comments
    // TODO Check this with the beta flag disabled (with the bounce page)
    // Remix is not including the X-Shopify-API-Request-Failure-Reauthorize-Url when throwing a Response
    // https://github.com/remix-run/remix/issues/5356
    throw new Response(undefined, {
      status: 401,
      statusText: 'Unauthorized',
      headers: redirectWithAppBridgeHeaders.getAppBridgeHeaders(url)
    });
  } else if (isEmbeddedRequest) {
    const params = new URLSearchParams({
      shop,
      host: requestUrl.searchParams.get('host'),
      exitIframe: url
    });
    throw serverRuntime.redirect(`${config.auth.exitIframePath}?${params.toString()}`);
  } else {
    // This will only ever happen for non-embedded apps, because the authenticator will stop before reaching this point
    throw serverRuntime.redirect(url);
  }
}

exports.requestBillingFactory = requestBillingFactory;
