'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var serverRuntime = require('@remix-run/server-runtime');
var shopifyApi = require('@shopify/shopify-api');
var cancel = require('../../billing/cancel.js');
var require$1 = require('../../billing/require.js');
var request = require('../../billing/request.js');
var appBridgeUrl = require('../helpers/app-bridge-url.js');
var addResponseHeaders = require('../helpers/add-response-headers.js');
var redirectToAuthPage = require('../helpers/redirect-to-auth-page.js');
var redirectWithExitiframe = require('../helpers/redirect-with-exitiframe.js');
var beginAuth = require('../helpers/begin-auth.js');
var ensureCorsHeaders = require('../helpers/ensure-cors-headers.js');
var validateSessionToken = require('../helpers/validate-session-token.js');
var getSessionTokenHeader = require('../helpers/get-session-token-header.js');
var rejectBotRequest = require('../helpers/reject-bot-request.js');
var respondToOptionsRequest = require('../helpers/respond-to-options-request.js');
var graphqlClient = require('./graphql-client.js');
var restClient = require('./rest-client.js');

const SESSION_TOKEN_PARAM = 'id_token';
class AuthStrategy {
  constructor({
    api,
    config,
    logger
  }) {
    this.api = void 0;
    this.config = void 0;
    this.logger = void 0;
    this.api = api;
    this.config = config;
    this.logger = logger;
  }
  async authenticateAdmin(request) {
    const {
      api,
      logger,
      config
    } = this;
    rejectBotRequest.rejectBotRequest({
      api,
      logger,
      config
    }, request);
    respondToOptionsRequest.respondToOptionsRequest({
      api,
      logger,
      config
    }, request);
    const cors = ensureCorsHeaders.ensureCORSHeadersFactory({
      api,
      logger,
      config
    }, request);
    let sessionContext;
    try {
      sessionContext = await this.authenticateAndGetSessionContext(request);
    } catch (errorOrResponse) {
      if (errorOrResponse instanceof Response) {
        cors(errorOrResponse);
      }
      throw errorOrResponse;
    }
    const context = {
      admin: this.createAdminApiContext(request, sessionContext.session),
      billing: this.createBillingContext(request, sessionContext.session),
      session: sessionContext.session,
      cors
    };
    if (config.isEmbeddedApp) {
      return {
        ...context,
        sessionToken: sessionContext.token
      };
    } else {
      return context;
    }
  }
  async authenticateAndGetSessionContext(request) {
    const {
      api,
      logger,
      config
    } = this;
    const url = new URL(request.url);
    const isPatchSessionToken = url.pathname === config.auth.patchSessionTokenPath;
    const isExitIframe = url.pathname === config.auth.exitIframePath;
    const isAuthRequest = url.pathname === config.auth.path;
    const isAuthCallbackRequest = url.pathname === config.auth.callbackPath;
    const sessionTokenHeader = getSessionTokenHeader.getSessionTokenHeader(request);
    logger.info('Authenticating admin request');
    if (isPatchSessionToken) {
      logger.debug('Rendering bounce page');
      throw this.renderAppBridge(request);
    } else if (isExitIframe) {
      const destination = url.searchParams.get('exitIframe');
      logger.debug('Rendering exit iframe page', {
        destination
      });
      throw this.renderAppBridge(request, destination);
    } else if (isAuthCallbackRequest) {
      throw await this.handleAuthCallbackRequest(request);
    } else if (isAuthRequest) {
      throw await this.handleAuthBeginRequest(request);
    } else if (sessionTokenHeader) {
      const sessionToken = await validateSessionToken.validateSessionToken({
        api,
        logger,
        config
      }, sessionTokenHeader);
      return this.validateAuthenticatedSession(request, sessionToken);
    } else {
      await this.validateUrlParams(request);
      await this.ensureInstalledOnShop(request);
      await this.ensureAppIsEmbeddedIfRequired(request);
      await this.ensureSessionTokenSearchParamIfRequired(request);
      return this.ensureSessionExists(request);
    }
  }
  async handleAuthBeginRequest(request) {
    const {
      api,
      config,
      logger
    } = this;
    logger.info('Handling OAuth begin request');
    const shop = this.ensureValidShopParam(request);
    logger.debug('OAuth request contained valid shop', {
      shop
    });

    // If we're loading from an iframe, we need to break out of it
    if (config.isEmbeddedApp && request.headers.get('Sec-Fetch-Dest') === 'iframe') {
      logger.debug('Auth request in iframe detected, exiting iframe', {
        shop
      });
      throw redirectWithExitiframe.redirectWithExitIframe({
        api,
        config,
        logger
      }, request, shop);
    } else {
      throw await beginAuth.beginAuth({
        api,
        config,
        logger
      }, request, false, shop);
    }
  }
  async handleAuthCallbackRequest(request) {
    const {
      api,
      config,
      logger
    } = this;
    logger.info('Handling OAuth callback request');
    const shop = this.ensureValidShopParam(request);
    try {
      const {
        session,
        headers: responseHeaders
      } = await api.auth.callback({
        rawRequest: request
      });
      await config.sessionStorage.storeSession(session);
      if (config.useOnlineTokens && !session.isOnline) {
        logger.info('Requesting online access token for offline session');
        await beginAuth.beginAuth({
          api,
          config,
          logger
        }, request, true, shop);
      }
      if (config.hooks.afterAuth) {
        logger.info('Running afterAuth hook');
        await config.hooks.afterAuth({
          session,
          admin: this.createAdminApiContext(request, session)
        });
      }
      throw await this.redirectToShopifyOrAppRoot(request, responseHeaders);
    } catch (error) {
      if (error instanceof Response) {
        throw error;
      }
      logger.error('Error during OAuth callback', {
        error: error.message
      });
      if (error instanceof shopifyApi.CookieNotFound) {
        throw await this.handleAuthBeginRequest(request);
      } else if (error instanceof shopifyApi.InvalidHmacError || error instanceof shopifyApi.InvalidOAuthError) {
        throw new Response(undefined, {
          status: 400,
          statusText: 'Invalid OAuth Request'
        });
      } else {
        throw new Response(undefined, {
          status: 500,
          statusText: 'Internal Server Error'
        });
      }
    }
  }
  async validateUrlParams(request) {
    const {
      api,
      config,
      logger
    } = this;
    if (config.isEmbeddedApp) {
      const url = new URL(request.url);
      const shop = api.utils.sanitizeShop(url.searchParams.get('shop'));
      if (!shop) {
        logger.debug('Missing or invalid shop, redirecting to login path', {
          shop
        });
        throw serverRuntime.redirect(config.auth.loginPath);
      }
      const host = api.utils.sanitizeHost(url.searchParams.get('host'));
      if (!host) {
        logger.debug('Invalid host, redirecting to login path', {
          host: url.searchParams.get('host')
        });
        throw serverRuntime.redirect(config.auth.loginPath);
      }
    }
  }
  async ensureInstalledOnShop(request) {
    const {
      api,
      config,
      logger
    } = this;
    const url = new URL(request.url);
    let shop = url.searchParams.get('shop');
    const isEmbedded = url.searchParams.get('embedded') === '1';

    // Ensure app is installed
    logger.debug('Ensuring app is installed on shop', {
      shop
    });
    const offlineId = shop ? api.session.getOfflineId(shop) : await api.session.getCurrentId({
      isOnline: false,
      rawRequest: request
    });
    if (!offlineId) {
      logger.info("Could not find a shop, can't authenticate request");
      throw new Response(undefined, {
        status: 400,
        statusText: 'Bad Request'
      });
    }
    const offlineSession = await config.sessionStorage.loadSession(offlineId);
    if (!offlineSession) {
      logger.info("Shop hasn't installed app yet, redirecting to OAuth", {
        shop
      });
      if (isEmbedded) {
        redirectWithExitiframe.redirectWithExitIframe({
          api,
          config,
          logger
        }, request, shop);
      } else {
        throw await beginAuth.beginAuth({
          api,
          config,
          logger
        }, request, false, shop);
      }
    }
    shop = shop || offlineSession.shop;
    if (config.isEmbeddedApp && !isEmbedded) {
      try {
        logger.debug('Ensuring offline session is valid before embedding', {
          shop
        });
        await this.testSession(offlineSession);
        logger.debug('Offline session is still valid, embedding app', {
          shop
        });
      } catch (error) {
        if (error instanceof shopifyApi.HttpResponseError) {
          if (error.response.code === 401) {
            logger.info('Shop session is no longer valid, redirecting to OAuth', {
              shop
            });
            throw await beginAuth.beginAuth({
              api,
              config,
              logger
            }, request, false, shop);
          } else {
            const message = JSON.stringify(error.response.body, null, 2);
            logger.error(`Unexpected error during session validation: ${message}`, {
              shop
            });
            throw new Response(undefined, {
              status: error.response.code,
              statusText: error.response.statusText
            });
          }
        } else if (error instanceof shopifyApi.GraphqlQueryError) {
          const context = {
            shop
          };
          if (error.response) {
            context.response = JSON.stringify(error.response);
          }
          logger.error(`Unexpected error during session validation: ${error.message}`, context);
          throw new Response(undefined, {
            status: 500,
            statusText: 'Internal Server Error'
          });
        }
      }
    }
  }
  async testSession(session) {
    const {
      api
    } = this;
    const client = new api.clients.Graphql({
      session
    });
    await client.query({
      data: `#graphql
        query shopifyAppShopName {
          shop {
            name
          }
        }
      `
    });
  }
  ensureValidShopParam(request) {
    const url = new URL(request.url);
    const {
      api
    } = this;
    const shop = api.utils.sanitizeShop(url.searchParams.get('shop'));
    if (!shop) {
      throw new Response('Shop param is invalid', {
        status: 400
      });
    }
    return shop;
  }
  async ensureAppIsEmbeddedIfRequired(request) {
    const {
      api,
      logger
    } = this;
    const url = new URL(request.url);
    const shop = url.searchParams.get('shop');
    if (api.config.isEmbeddedApp && url.searchParams.get('embedded') !== '1') {
      logger.debug('App is not embedded, redirecting to Shopify', {
        shop
      });
      await this.redirectToShopifyOrAppRoot(request);
    }
  }
  async ensureSessionTokenSearchParamIfRequired(request) {
    const {
      api,
      logger
    } = this;
    const url = new URL(request.url);
    const shop = url.searchParams.get('shop');
    const searchParamSessionToken = url.searchParams.get(SESSION_TOKEN_PARAM);
    if (api.config.isEmbeddedApp && !searchParamSessionToken) {
      logger.debug('Missing session token in search params, going to bounce page', {
        shop
      });
      this.redirectToBouncePage(url);
    }
  }
  async ensureSessionExists(request) {
    const {
      api,
      config,
      logger
    } = this;
    const url = new URL(request.url);
    const shop = url.searchParams.get('shop');
    const searchParamSessionToken = url.searchParams.get(SESSION_TOKEN_PARAM);
    if (api.config.isEmbeddedApp) {
      logger.debug('Session token is present in query params, validating session', {
        shop
      });
      const sessionToken = await validateSessionToken.validateSessionToken({
        api,
        config,
        logger
      }, searchParamSessionToken);
      return this.validateAuthenticatedSession(request, sessionToken);
    } else {
      // eslint-disable-next-line no-warning-comments
      // TODO move this check into loadSession once we add support for it in the library
      // https://github.com/orgs/Shopify/projects/6899/views/1?pane=issue&itemId=28378114
      const sessionId = await api.session.getCurrentId({
        isOnline: config.useOnlineTokens,
        rawRequest: request
      });
      if (!sessionId) {
        logger.debug('Session id not found in cookies, redirecting to OAuth', {
          shop
        });
        throw await beginAuth.beginAuth({
          api,
          config,
          logger
        }, request, false, shop);
      }
      return {
        session: await this.loadSession(request, shop, sessionId)
      };
    }
  }
  async validateAuthenticatedSession(request, payload) {
    const {
      config,
      logger,
      api
    } = this;
    const dest = new URL(payload.dest);
    const shop = dest.hostname;
    const sessionId = config.useOnlineTokens ? api.session.getJwtSessionId(shop, payload.sub) : api.session.getOfflineId(shop);
    const session = await this.loadSession(request, shop, sessionId);
    logger.debug('Found session, request is valid', {
      shop
    });
    return {
      session,
      token: payload
    };
  }
  async loadSession(request, shop, sessionId) {
    const {
      api,
      config,
      logger
    } = this;
    logger.debug('Loading session from storage', {
      sessionId
    });
    const session = await config.sessionStorage.loadSession(sessionId);
    if (!session) {
      logger.debug('No session found, redirecting to OAuth', {
        shop
      });
      await redirectToAuthPage.redirectToAuthPage({
        api,
        config,
        logger
      }, request, shop);
    } else if (!session.isActive(config.scopes)) {
      logger.debug('Found a session, but it has expired, redirecting to OAuth', {
        shop
      });
      await redirectToAuthPage.redirectToAuthPage({
        api,
        config,
        logger
      }, request, shop);
    }
    return session;
  }
  async redirectToShopifyOrAppRoot(request, responseHeaders) {
    const {
      api
    } = this;
    const url = new URL(request.url);
    const host = api.utils.sanitizeHost(url.searchParams.get('host'));
    const shop = api.utils.sanitizeShop(url.searchParams.get('shop'));
    const redirectUrl = api.config.isEmbeddedApp ? await api.auth.getEmbeddedAppUrl({
      rawRequest: request
    }) : `/?shop=${shop}&host=${encodeURIComponent(host)}`;
    throw serverRuntime.redirect(redirectUrl, {
      headers: responseHeaders
    });
  }
  redirectToBouncePage(url) {
    const {
      api,
      config
    } = this;

    // eslint-disable-next-line no-warning-comments
    // TODO this is to work around a remix bug
    // https://github.com/orgs/Shopify/projects/6899/views/1?pane=issue&itemId=28376650
    url.protocol = `${api.config.hostScheme}:`;
    const params = new URLSearchParams(url.search);
    params.set('shopify-reload', url.href);

    // eslint-disable-next-line no-warning-comments
    // TODO Make sure this works on chrome without a tunnel (weird HTTPS redirect issue)
    // https://github.com/orgs/Shopify/projects/6899/views/1?pane=issue&itemId=28376650
    throw serverRuntime.redirect(`${config.auth.patchSessionTokenPath}?${params.toString()}`);
  }
  renderAppBridge(request, redirectTo) {
    const {
      config
    } = this;
    let redirectToScript = '';
    if (redirectTo) {
      const redirectUrl = decodeURIComponent(redirectTo.startsWith('/') ? `${config.appUrl}${redirectTo}` : redirectTo);
      redirectToScript = `<script>window.open("${redirectUrl}", "_top")</script>`;
    }
    const responseHeaders = new Headers({
      'content-type': 'text/html;charset=utf-8'
    });
    addResponseHeaders.addDocumentResponseHeaders(responseHeaders, config.isEmbeddedApp, new URL(request.url).searchParams.get('shop'));
    throw new Response(`
        <script data-api-key="${config.apiKey}" src="${appBridgeUrl.appBridgeUrl()}"></script>
        ${redirectToScript}
      `, {
      headers: responseHeaders
    });
  }
  overriddenRestClient(request, session) {
    const {
      api,
      config,
      logger
    } = this;
    const client = new restClient.RemixRestClient({
      params: {
        api,
        config,
        logger
      },
      request,
      session
    });
    if (api.rest) {
      client.resources = {};
      const RestResourceClient = restClient.restResourceClientFactory({
        params: {
          api,
          config,
          logger
        },
        request,
        session
      });
      Object.entries(api.rest).forEach(([name, resource]) => {
        class RemixResource extends resource {}
        RemixResource.Client = RestResourceClient;
        Reflect.defineProperty(RemixResource, 'name', {
          value: name
        });
        Reflect.set(client.resources, name, RemixResource);
      });
    }
    return client;
  }
  createBillingContext(request$1, session) {
    const {
      api,
      logger,
      config
    } = this;
    return {
      require: require$1.requireBillingFactory({
        api,
        logger,
        config
      }, request$1, session),
      request: request.requestBillingFactory({
        api,
        logger,
        config
      }, request$1, session),
      cancel: cancel.cancelBillingFactory({
        api,
        logger,
        config
      }, request$1, session)
    };
  }
  createAdminApiContext(request, session) {
    const {
      api,
      config,
      logger
    } = this;
    return {
      rest: this.overriddenRestClient(request, session),
      graphql: graphqlClient.graphqlClientFactory({
        params: {
          api,
          config,
          logger
        },
        request,
        session
      })
    };
  }
}

exports.AuthStrategy = AuthStrategy;
