import { UnicodeCharacterSet } from '../constants/index.mjs';
import { identifyScripts } from './identifyScripts.mjs';

function tryAbbreviateName({
  firstName,
  lastName,
  idealMaxLength = 3
}) {
  if (!firstName && !lastName) {
    return undefined;
  }

  const firstNameTrimmed = firstName === null || firstName === void 0 ? void 0 : firstName.trim();
  const lastNameTrimmed = lastName === null || lastName === void 0 ? void 0 : lastName.trim();
  const combinedName = [firstNameTrimmed, lastNameTrimmed].join('');

  if (new RegExp(`${UnicodeCharacterSet.Punctuation}|\\s`).test(combinedName)) {
    return undefined;
  }

  const scripts = identifyScripts(combinedName);

  if (scripts.length !== 1) {
    return undefined;
  }

  const script = scripts[0];

  switch (script) {
    case UnicodeCharacterSet.Latin:
      return [firstNameTrimmed === null || firstNameTrimmed === void 0 ? void 0 : firstNameTrimmed[0], lastNameTrimmed === null || lastNameTrimmed === void 0 ? void 0 : lastNameTrimmed[0]].join('');

    case UnicodeCharacterSet.Han:
    case UnicodeCharacterSet.Katakana:
    case UnicodeCharacterSet.Hiragana:
      return lastNameTrimmed;

    case UnicodeCharacterSet.Hangul:
      if (firstNameTrimmed) {
        if (firstNameTrimmed.length > idealMaxLength) {
          var _getGraphemes;

          return (_getGraphemes = getGraphemes({
            text: firstNameTrimmed,
            locale: 'ko'
          })) === null || _getGraphemes === void 0 ? void 0 : _getGraphemes[0];
        } else {
          return firstNameTrimmed;
        }
      } else {
        return lastNameTrimmed;
      }

    case UnicodeCharacterSet.Thai:
      if (firstNameTrimmed) {
        var _getGraphemes2;

        return (_getGraphemes2 = getGraphemes({
          text: firstNameTrimmed,
          locale: 'th'
        })) === null || _getGraphemes2 === void 0 ? void 0 : _getGraphemes2[0];
      } else {
        var _getGraphemes3;

        return (_getGraphemes3 = getGraphemes({
          text: lastNameTrimmed,
          locale: 'th'
        })) === null || _getGraphemes3 === void 0 ? void 0 : _getGraphemes3[0];
      }

    default:
      return undefined;
  }
}

function getGraphemes({
  text,
  locale
}) {
  // returns undefined when Intl.Segmenter does not exist in our JS environment (such as in Firefox)
  if (!text || !Intl.Segmenter) {
    return undefined;
  }

  const segmenter = new Intl.Segmenter(locale, {
    granularity: 'grapheme'
  });
  return Array.from(segmenter.segment(text)).map(grapheme => grapheme.segment);
}

export { tryAbbreviateName };
