import { formatDate, isFutureDate, isLessThanOneMinuteAgo, isLessThanOneHourAgo, TimeUnit, isToday, isYesterday, isLessThanOneWeekAgo, isLessThanOneYearAgo, isTomorrow, isLessThanOneWeekAway, isLessThanOneYearAway } from '@shopify/dates';
import { memoize } from '@shopify/function-enhancers';
import { languageFromLocale, regionFromLocale } from '@shopify/i18n';
import { LanguageDirection } from './types.mjs';
import { RTL_LANGUAGES, dateStyle, DateStyle, WEEK_START_DAYS, DEFAULT_WEEK_START_DAY, EASTERN_NAME_ORDER_FORMATTERS, UnicodeCharacterSet, CurrencyShortFormException } from './constants/index.mjs';
import { MissingCurrencyCodeError, MissingCountryError } from './errors.mjs';
import { translate, getTranslationTree, memoizedNumberFormatter, memoizedPluralRules } from './utilities/translate.mjs';
import { currencyDecimalPlaces, DEFAULT_DECIMAL_PLACES } from './constants/currency-decimal-places.mjs';
import { tryAbbreviateName } from './utilities/tryAbbreviateName.mjs';
import { identifyScripts } from './utilities/identifyScripts.mjs';
import { getCurrencySymbol } from './utilities/money.mjs';
import { convertFirstSpaceToNonBreakingSpace } from './utilities/string.mjs';

// Used for currencies that don't use fractional units (eg. JPY)
const PERIOD = '.';
const NEGATIVE_SIGN = '-';
const REGEX_DIGITS = /\d/g;
const REGEX_NON_DIGITS = /\D/g;
const REGEX_PERIODS = /\./g;
class I18n {
  get language() {
    return languageFromLocale(this.locale);
  }

  get region() {
    return regionFromLocale(this.locale);
  }
  /**
   * @deprecated Use I18n#region instead.
   */


  get countryCode() {
    return regionFromLocale(this.locale);
  }

  get languageDirection() {
    return RTL_LANGUAGES.includes(this.language) ? LanguageDirection.Rtl : LanguageDirection.Ltr;
  }

  get isRtlLanguage() {
    return this.languageDirection === LanguageDirection.Rtl;
  }

  get isLtrLanguage() {
    return this.languageDirection === LanguageDirection.Ltr;
  }

  constructor(translations, {
    locale: _locale,
    currency: _currency,
    timezone,
    country,
    pseudolocalize = false,
    onError,
    loading,
    interpolate
  }) {
    this.getCurrencySymbol = (currencyCode, locale = this.locale) => {
      const currency = currencyCode || this.defaultCurrency;

      if (currency == null) {
        throw new MissingCurrencyCodeError('formatCurrency cannot be called without a currency code.');
      }

      return this.getShortCurrencySymbol(currency, locale);
    };

    this.numberSymbols = memoize(() => {
      const formattedNumber = this.formatNumber(123456.7, {
        maximumFractionDigits: 1,
        minimumFractionDigits: 1
      });
      let thousandSymbol;
      let decimalSymbol;

      for (const char of formattedNumber) {
        if (isNaN(parseInt(char, 10))) {
          if (thousandSymbol) decimalSymbol = char;else thousandSymbol = char;
        }
      }

      return {
        thousandSymbol,
        decimalSymbol
      };
    });
    this.translations = translations;
    this.locale = _locale;
    this.defaultCountry = country;
    this.defaultCurrency = _currency;
    this.defaultTimezone = timezone;
    this.pseudolocalize = pseudolocalize;
    this.defaultInterpolate = interpolate;
    this.onError = onError || this.defaultOnError;
    this.loading = loading || false;
  }

  translate(id, optionsOrReplacements, replacements) {
    const {
      pseudolocalize,
      defaultInterpolate
    } = this;
    let normalizedOptions;
    const defaultOptions = {
      pseudotranslate: pseudolocalize,
      interpolate: defaultInterpolate
    };

    if (optionsOrReplacements == null) {
      normalizedOptions = defaultOptions;
    } else if (this.isTranslateOptions(optionsOrReplacements)) {
      normalizedOptions = { ...defaultOptions,
        ...optionsOrReplacements,
        replacements
      };
    } else {
      normalizedOptions = { ...defaultOptions,
        replacements: optionsOrReplacements
      };
    }

    try {
      return translate(id, normalizedOptions, this.translations, this.locale);
    } catch (error) {
      this.onError(error);
      return '';
    }
  }

  getTranslationTree(id, replacements) {
    try {
      if (!replacements) {
        return getTranslationTree(id, this.translations, this.locale);
      }

      return getTranslationTree(id, this.translations, this.locale, replacements);
    } catch (error) {
      this.onError(error);
      return '';
    }
  }

  translationKeyExists(id) {
    try {
      getTranslationTree(id, this.translations, this.locale);
      return true;
    } catch (error) {
      return false;
    }
  }

  formatNumber(amount, {
    as,
    precision,
    ...options
  } = {}) {
    const {
      locale,
      defaultCurrency: currency
    } = this;

    if (as === 'currency' && currency == null && options.currency == null) {
      this.onError(new MissingCurrencyCodeError(`formatNumber(amount, {as: 'currency'}) cannot be called without a currency code.`));
      return '';
    }

    return memoizedNumberFormatter(locale, {
      style: as,
      maximumFractionDigits: precision,
      currency,
      ...options
    }).format(amount);
  }

  unformatNumber(input) {
    const {
      decimalSymbol
    } = this.numberSymbols();
    const normalizedValue = this.normalizedNumber(input, decimalSymbol);
    return normalizedValue === '' ? '' : parseFloat(normalizedValue).toString();
  }

  formatCurrency(amount, {
    form,
    ...options
  } = {}) {
    switch (form) {
      case 'auto':
        return this.formatCurrencyAuto(amount, options);

      case 'explicit':
        return this.formatCurrencyExplicit(amount, options);

      case 'short':
        return this.formatCurrencyShort(amount, options);

      case 'none':
        return this.formatCurrencyNone(amount, options);

      default:
        return this.formatCurrencyAuto(amount, options);
    }
  }

  unformatCurrency(input, currencyCode) {
    const {
      decimalSymbol
    } = this.numberSymbols();
    const decimalPlaces = currencyDecimalPlaces.get(currencyCode.toUpperCase());
    const normalizedValue = this.normalizedNumber(input, decimalSymbol, decimalPlaces);

    if (normalizedValue === '') {
      return '';
    }

    if (decimalPlaces === 0) {
      const roundedAmount = parseFloat(normalizedValue).toFixed(0);
      return `${roundedAmount}.${'0'.repeat(DEFAULT_DECIMAL_PLACES)}`;
    }

    return parseFloat(normalizedValue).toFixed(decimalPlaces);
  }

  formatPercentage(amount, options = {}) {
    return this.formatNumber(amount, {
      as: 'percent',
      ...options
    });
  }

  formatDate(date, options = {}) {
    const {
      locale,
      defaultTimezone
    } = this;
    const {
      timeZone = defaultTimezone
    } = options;
    const {
      style = undefined,
      ...formatOptions
    } = options || {};

    if (style) {
      switch (style) {
        case DateStyle.Humanize:
          return this.humanizeDate(date, { ...formatOptions,
            timeZone
          });

        case DateStyle.DateTime:
          return this.formatDateTime(date, { ...formatOptions,
            timeZone,
            ...dateStyle[style]
          });

        default:
          return this.formatDate(date, { ...formatOptions,
            ...dateStyle[style]
          });
      }
    }

    return formatDate(date, locale, { ...formatOptions,
      timeZone
    });
  }

  ordinal(amount) {
    const {
      locale
    } = this;
    const group = memoizedPluralRules(locale, {
      type: 'ordinal'
    }).select(amount);
    return this.translate(group, {
      scope: 'ordinal'
    }, {
      amount
    });
  }

  weekStartDay(argCountry) {
    const country = argCountry || this.defaultCountry;

    if (!country) {
      throw new MissingCountryError('weekStartDay() cannot be called without a country code.');
    }

    return WEEK_START_DAYS.get(country) || DEFAULT_WEEK_START_DAY;
  }

  /**
   * @deprecated Replace usage of `i18n.getCurrencySymbolLocalized(locale, currency)` with `i18n.getCurrencySymbol(currency, locale)`
   */
  getCurrencySymbolLocalized(locale, currency) {
    return this.getShortCurrencySymbol(currency, locale);
  }

  formatName(firstName, lastName, options) {
    if (!firstName) {
      return lastName || '';
    }

    if (!lastName) {
      return firstName;
    }

    const isFullName = Boolean(options && options.full);
    const customNameFormatter = EASTERN_NAME_ORDER_FORMATTERS.get(this.locale) || EASTERN_NAME_ORDER_FORMATTERS.get(this.language);

    if (customNameFormatter) {
      return customNameFormatter(firstName, lastName, isFullName);
    }

    if (isFullName) {
      return `${firstName} ${lastName}`;
    }

    return firstName;
  } // Note: A similar Ruby implementation of this function also exists at https://github.com/Shopify/shopify-i18n/blob/main/lib/shopify-i18n/name_formatter.rb.


  abbreviateName({
    firstName,
    lastName,
    idealMaxLength = 3
  }) {
    var _tryAbbreviateName;

    return (_tryAbbreviateName = tryAbbreviateName({
      firstName,
      lastName,
      idealMaxLength
    })) !== null && _tryAbbreviateName !== void 0 ? _tryAbbreviateName : this.formatName(firstName, lastName);
  }

  identifyScript(text) {
    return identifyScripts(text);
  }

  hasEasternNameOrderFormatter() {
    const easternNameOrderFormatter = EASTERN_NAME_ORDER_FORMATTERS.get(this.locale) || EASTERN_NAME_ORDER_FORMATTERS.get(this.language);
    return Boolean(easternNameOrderFormatter);
  } // eslint-disable-next-line @typescript-eslint/member-ordering


  formatCurrencyAuto(amount, options = {}) {
    // use the short format if we can't determine a currency match, or if the
    // currencies match, use explicit when the currencies definitively do not
    // match.
    const formatShort = options.currency == null || this.defaultCurrency == null || options.currency === this.defaultCurrency;
    return formatShort ? this.formatCurrencyShort(amount, options) : this.formatCurrencyExplicit(amount, options);
  }

  formatCurrencyExplicit(amount, options = {}) {
    const value = this.formatCurrencyShort(amount, options);
    const isoCode = options.currency || this.defaultCurrency || '';

    if (value.includes(isoCode)) {
      return value;
    }

    return `${value} ${isoCode}`;
  }

  formatCurrencyShort(amount, options = {}) {
    var _negativeRegex$exec;

    const formattedAmount = this.formatCurrencyNone(amount, options);
    const negativeRegex = new RegExp(`${UnicodeCharacterSet.DirectionControl}*${UnicodeCharacterSet.Negative}`, 'g');
    const negativeMatch = ((_negativeRegex$exec = negativeRegex.exec(formattedAmount)) === null || _negativeRegex$exec === void 0 ? void 0 : _negativeRegex$exec.shift()) || '';
    const shortSymbol = this.getShortCurrencySymbol(options.currency);
    const formattedWithSymbol = shortSymbol.prefixed ? `${shortSymbol.symbol}${formattedAmount}` : `${formattedAmount}${shortSymbol.symbol}`;
    return `${negativeMatch}${formattedWithSymbol.replace(negativeMatch, '')}`;
  }

  formatCurrencyNone(amount, options = {}) {
    const {
      locale
    } = this;
    let adjustedPrecision = options.precision;

    if (adjustedPrecision === undefined) {
      const currency = options.currency || this.defaultCurrency || '';
      adjustedPrecision = currencyDecimalPlaces.get(currency.toUpperCase());
    }

    return memoizedNumberFormatter(locale, {
      style: 'decimal',
      minimumFractionDigits: adjustedPrecision,
      maximumFractionDigits: adjustedPrecision,
      ...options
    }).format(amount);
  } // Intl.NumberFormat sometimes annotates the "currency symbol" with a country code.
  // For example, in locale 'fr-FR', 'USD' is given the "symbol" of " $US".
  // This method strips out the country-code annotation, if there is one.
  // (So, for 'fr-FR' and 'USD', the return value would be " $").
  //
  // For other currencies, e.g. CHF and OMR, the "symbol" is the ISO currency code.
  // In those cases, we return the full currency code without stripping the country.


  getShortCurrencySymbol(currency = this.defaultCurrency || '', locale = this.locale) {
    const regionCode = currency.substring(0, 2);
    let shortSymbolResult; // currencyDisplay: 'narrowSymbol' was added to iOS in v14.5. See https://caniuse.com/?search=currencydisplay
    // We still support ios 12/13, so we need to check if this works and fallback to the default if not
    // All other supported browsers understand narrowSymbol, so once our minimum iOS version is updated we can remove this fallback

    try {
      shortSymbolResult = getCurrencySymbol(locale, {
        currency,
        currencyDisplay: 'narrowSymbol'
      });
    } catch {
      shortSymbolResult = getCurrencySymbol(locale, {
        currency
      });
    }

    if (currency in CurrencyShortFormException) {
      return {
        symbol: CurrencyShortFormException[currency],
        prefixed: shortSymbolResult.prefixed
      };
    }

    const shortSymbol = shortSymbolResult.symbol.replace(regionCode, '');
    const alphabeticCharacters = /[A-Za-zÀ-ÖØ-öø-ÿĀ-ɏḂ-ỳ]/;
    return alphabeticCharacters.exec(shortSymbol) ? shortSymbolResult : {
      symbol: shortSymbol,
      prefixed: shortSymbolResult.prefixed
    };
  }

  humanizeDate(date, options) {
    return isFutureDate(date) ? this.humanizeFutureDate(date, options) : this.humanizePastDate(date, options);
  }

  formatDateTime(date, options) {
    const {
      defaultTimezone
    } = this;
    const {
      timeZone = defaultTimezone
    } = options;
    return this.translate('date.humanize.lessThanOneYearAway', {
      date: this.getDateFromDate(date, { ...options,
        timeZone
      }),
      time: this.getTimeFromDate(date, { ...options,
        timeZone
      })
    });
  }

  humanizePastDate(date, options) {
    if (isLessThanOneMinuteAgo(date)) {
      return this.translate('date.humanize.lessThanOneMinuteAgo');
    }

    if (isLessThanOneHourAgo(date)) {
      const now = new Date();
      const minutes = Math.floor((now.getTime() - date.getTime()) / TimeUnit.Minute);
      return this.translate('date.humanize.lessThanOneHourAgo', {
        count: minutes
      });
    }

    const timeZone = options === null || options === void 0 ? void 0 : options.timeZone;
    const time = this.getTimeFromDate(date, options);

    if (isToday(date, timeZone)) {
      return time;
    }

    if (isYesterday(date, timeZone)) {
      return this.translate('date.humanize.yesterday', {
        time
      });
    }

    if (isLessThanOneWeekAgo(date)) {
      const weekday = this.getWeekdayFromDate(date, options);
      return this.translate('date.humanize.lessThanOneWeekAgo', {
        weekday,
        time
      });
    }

    if (isLessThanOneYearAgo(date)) {
      const monthDay = this.getMonthDayFromDate(date, options);
      return this.translate('date.humanize.lessThanOneYearAgo', {
        date: monthDay,
        time
      });
    }

    return this.formatDate(date, { ...options,
      style: DateStyle.Short
    });
  }

  humanizeFutureDate(date, options) {
    const timeZone = options === null || options === void 0 ? void 0 : options.timeZone;
    const time = this.getTimeFromDate(date, options);

    if (isToday(date, timeZone)) {
      return this.translate('date.humanize.today', {
        time
      });
    }

    if (isTomorrow(date, timeZone)) {
      return this.translate('date.humanize.tomorrow', {
        time
      });
    }

    if (isLessThanOneWeekAway(date)) {
      const weekday = this.getWeekdayFromDate(date, options);
      return this.translate('date.humanize.lessThanOneWeekAway', {
        weekday,
        time
      });
    }

    if (isLessThanOneYearAway(date)) {
      const monthDay = this.getMonthDayFromDate(date, options);
      return this.translate('date.humanize.lessThanOneYearAway', {
        date: monthDay,
        time
      });
    }

    return this.formatDate(date, { ...options,
      style: DateStyle.Short
    });
  }

  getTimeZone(date, options) {
    const {
      localeMatcher,
      formatMatcher,
      timeZone
    } = options || {};
    const hourZone = this.formatDate(date, {
      localeMatcher,
      formatMatcher,
      timeZone,
      hour12: false,
      timeZoneName: 'short',
      hour: 'numeric'
    });
    const zoneMatchGroup = /\s([\w()+\-:.]+$)/.exec(hourZone);
    return zoneMatchGroup ? zoneMatchGroup[1] : '';
  }

  getDateFromDate(date, options) {
    const {
      localeMatcher,
      formatMatcher,
      weekday,
      day,
      month,
      year,
      era,
      timeZone,
      timeZoneName
    } = options || {};
    const formattedDate = this.formatDate(date, {
      localeMatcher,
      formatMatcher,
      weekday,
      day,
      month,
      year,
      era,
      timeZone,
      timeZoneName: timeZoneName === 'short' ? undefined : timeZoneName
    });
    return formattedDate;
  }

  getTimeFromDate(date, options) {
    const {
      localeMatcher,
      formatMatcher,
      hour12,
      timeZone,
      timeZoneName
    } = options || {};
    const formattedTime = this.formatDate(date, {
      localeMatcher,
      formatMatcher,
      hour12,
      timeZone,
      timeZoneName: timeZoneName === 'short' ? undefined : timeZoneName,
      hour: 'numeric',
      minute: '2-digit'
    }).toLocaleLowerCase();
    const time = timeZoneName === 'short' ? `${formattedTime} ${this.getTimeZone(date, options)}` : formattedTime;
    return convertFirstSpaceToNonBreakingSpace(time);
  }

  getWeekdayFromDate(date, options) {
    const {
      localeMatcher,
      formatMatcher,
      hour12,
      timeZone
    } = options || {};
    return this.formatDate(date, {
      localeMatcher,
      formatMatcher,
      hour12,
      timeZone,
      weekday: 'long'
    });
  }

  getMonthDayFromDate(date, options) {
    const {
      localeMatcher,
      formatMatcher,
      hour12,
      timeZone
    } = options || {};
    return this.formatDate(date, {
      localeMatcher,
      formatMatcher,
      hour12,
      timeZone,
      month: 'short',
      day: 'numeric'
    });
  }

  normalizedNumber(input, decimalSymbol, decimalPlaces = DEFAULT_DECIMAL_PLACES) {
    const maximumDecimalPlaces = Math.max(decimalPlaces, DEFAULT_DECIMAL_PLACES);
    const lastIndexOfPeriod = input.lastIndexOf(PERIOD);
    let lastIndexOfDecimal = input.lastIndexOf(decimalSymbol); // For locales that do not use period as the decimal symbol, users may still input a period
    // and expect it to be treated as the decimal symbol for their locale.

    if (decimalSymbol !== PERIOD && (input.match(REGEX_PERIODS) || []).length === 1 && this.decimalValue(input, lastIndexOfPeriod).length <= maximumDecimalPlaces) {
      lastIndexOfDecimal = lastIndexOfPeriod;
    }

    const integerValue = this.integerValue(input, lastIndexOfDecimal);
    const decimalValue = this.decimalValue(input, lastIndexOfDecimal);
    const negativeRegex = new RegExp(`^(${UnicodeCharacterSet.DirectionControl}|\\s)*${UnicodeCharacterSet.Negative}`, 'u');
    const negativeSign = input.match(negativeRegex) ? NEGATIVE_SIGN : '';
    const normalizedDecimal = lastIndexOfDecimal === -1 ? '' : PERIOD;
    const normalizedValue = `${negativeSign}${integerValue}${normalizedDecimal}${decimalValue}`;
    return normalizedValue.match(REGEX_DIGITS) ? normalizedValue : '';
  }

  integerValue(input, lastIndexOfDecimal) {
    return input.substring(0, lastIndexOfDecimal).replace(REGEX_NON_DIGITS, '');
  }

  decimalValue(input, lastIndexOfDecimal) {
    return input.substring(lastIndexOfDecimal + 1).replace(REGEX_NON_DIGITS, '');
  }

  isTranslateOptions(object) {
    return 'scope' in object;
  }

  defaultOnError(error) {
    throw error;
  }

}

export { I18n };
