'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var React = require('react');
var reactHooks = require('@shopify/react-hooks');
var i18n = require('./i18n.js');
var context = require('./context.js');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var React__default = /*#__PURE__*/_interopDefaultLegacy(React);

function useI18n(options) {
  const manager = React__default["default"].useContext(context.I18nContext);

  if (manager == null) {
    throw new Error('Missing i18n manager. Make sure to use an <I18nContext.Provider /> somewhere in your React tree.');
  }

  const registerOptions = React__default["default"].useRef(options);

  if (shouldRegister(registerOptions.current) !== shouldRegister(options)) {
    throw new Error('You switched between providing registration options and not providing them, which is not supported.');
  } // Yes, this would usually be dangerous. But just above this line, we check to make
  // sure that they never switch from the case where `options == null` to `options != null`,
  // so we know that a given use of this hook will only ever hit one of these two cases.

  /* eslint-disable react-hooks/rules-of-hooks */


  if (options == null) {
    return useSimpleI18n(manager);
  } else {
    return useComplexI18n(options, manager);
  }
  /* eslint-enable react-hooks/rules-of-hooks */

}

function useComplexI18n({
  id,
  fallback,
  translations
}, manager) {
  const managerRef = React__default["default"].useRef(null);
  const unsubscribeRef = React__default["default"].useRef(noop);
  const parentIds = React__default["default"].useContext(context.I18nIdsContext); // Parent IDs can only change when a parent gets added/ removed,
  // which would cause the component using `useI18n` to unmount.
  // We also don't support the `id` changing between renders. For these
  // reasons, it's safe to just store the IDs once and never let them change.

  const ids = reactHooks.useLazyRef(() => id ? [id, ...parentIds] : parentIds); // When the manager changes, we need to do the following IMMEDIATELY (i.e.,
  // not in a useEffect callback):
  //
  // 1. Register the component’s translations. This ensures that the first render gets
  //    the synchronous translations, if available.
  // 2. Unsubscribe from changes to a previous manager.
  // 3. Subscribe to changes from the new manager. This ensures that if the subscription
  //    is updated between render and `useEffect`, the state update is not lost.

  if (manager !== managerRef.current) {
    managerRef.current = manager;
    unsubscribeRef.current();
    unsubscribeRef.current = manager.subscribe(ids.current, ({
      translations,
      loading
    }, details) => {
      const newI18n = new i18n.I18n(translations, { ...details,
        loading
      });
      i18nRef.current = newI18n;
      setI18n(newI18n);
    });

    if (id && (translations || fallback)) {
      manager.register({
        id,
        translations,
        fallback
      });
    }
  }

  const [i18n$1, setI18n] = React__default["default"].useState(() => {
    const managerState = manager.state(ids.current);
    const {
      translations,
      loading
    } = managerState;
    return new i18n.I18n(translations, { ...manager.details,
      loading
    });
  });
  const i18nRef = React__default["default"].useRef(i18n$1);
  React__default["default"].useEffect(() => {
    return unsubscribeRef.current;
  }, []); // We use refs in this component so that it never changes. If this component
  // is regenerated, it will unmount the entire tree of the previous component,
  // which is usually not desirable. Technically, this does leave surface area
  // for a bug to sneak in: if the component that renders this does so inside
  // a component that blocks the update from passing down, nothing will force
  // this component to re-render, so no descendants will get the new ids/ i18n
  // value. Because we don't actually have any such cases, we're OK with this
  // for now.

  const shareTranslationsComponent = reactHooks.useLazyRef(() => function ShareTranslations({
    children
  }) {
    return /*#__PURE__*/React__default["default"].createElement(context.I18nIdsContext.Provider, {
      value: ids.current
    }, /*#__PURE__*/React__default["default"].createElement(context.I18nParentContext.Provider, {
      value: i18nRef.current
    }, children));
  });
  return [i18n$1, shareTranslationsComponent.current];
}

function useSimpleI18n(manager) {
  const i18n$1 = React__default["default"].useContext(context.I18nParentContext) || new i18n.I18n([], manager.details);
  return [i18n$1, IdentityComponent];
}

function IdentityComponent({
  children
}) {
  return /*#__PURE__*/React__default["default"].createElement(React__default["default"].Fragment, null, children);
}

function shouldRegister({
  fallback,
  translations
} = {}) {
  return fallback != null || translations != null;
}

function noop() {}

exports.useI18n = useI18n;
