'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var path = require('path');
var fs = require('fs-extra');
var cloneDeep = require('lodash.clonedeep');
var merge = require('lodash.merge');
var stripJsonComments = require('strip-json-comments');
var shared = require('./shared.js');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var fs__default = /*#__PURE__*/_interopDefaultLegacy(fs);
var cloneDeep__default = /*#__PURE__*/_interopDefaultLegacy(cloneDeep);
var merge__default = /*#__PURE__*/_interopDefaultLegacy(merge);
var stripJsonComments__default = /*#__PURE__*/_interopDefaultLegacy(stripJsonComments);

async function generateTranslationDictionaries(locales, {
  fallbackLocale = shared.DEFAULT_FALLBACK_LOCALE,
  rootDir = process.cwd()
} = {}) {
  if (!locales || locales.length === 0) {
    throw new Error('generateTranslationDictionary must be called with at least one locale.');
  }

  const translationBuckets = shared.findTranslationBuckets(rootDir);
  await Promise.all(Object.entries(translationBuckets).map(async ([translationsDir, translationFilePaths]) => {
    const fallbackTranslations = await readLocaleTranslations(fallbackLocale, translationFilePaths);
    const dictionary = await locales.reduce(async (accPromise, locale) => {
      const localeTranslations = await readLocaleTranslations(locale, translationFilePaths);
      const acc = await accPromise;
      acc[locale] = merge__default["default"](cloneDeep__default["default"](fallbackTranslations), localeTranslations);
      return acc;
    }, {});
    const contentStr = JSON.stringify(dictionary); // Writing the content out as a JSON.parse-wrapped string seems
    // counter-intuitive, but browsers can parse this faster than they
    // can parse JavaScript ‾\_(ツ)_/‾
    // https://www.youtube.com/watch?v=ff4fgQxPaO0

    await fs__default["default"].writeFile(path.join(translationsDir, 'index.js'), `export default JSON.parse(${JSON.stringify(contentStr)})`);
  }));
}

async function readLocaleTranslations(locale, translationFilePaths) {
  const translationPath = translationFilePaths.find(filePath => filePath.endsWith(`/${locale}.json`));
  return translationPath ? JSON.parse(stripJsonComments__default["default"](await fs__default["default"].readFile(translationPath, 'utf-8'))) : {};
}

exports.generateTranslationDictionaries = generateTranslationDictionaries;
