import React from 'react';
import { renderToStaticMarkup } from 'react-dom/server';
import { EffectContext } from './context.mjs';
import { EffectManager } from './manager.mjs';
export { Effect } from './Effect.mjs';

const DEFAULT_MAX_PASSES = 5;
function extract(app, {
  include,
  maxPasses = DEFAULT_MAX_PASSES,
  decorate = identity,
  renderFunction = renderToStaticMarkup,
  betweenEachPass,
  afterEachPass
} = {}) {
  // eslint-disable-next-line react/jsx-no-constructed-context-values
  const manager = new EffectManager({
    include
  });
  const element = /*#__PURE__*/React.createElement(EffectContext.Provider, {
    value: manager
  }, decorate(app));
  return async function perform(index = 0) {
    const start = Date.now();
    const result = renderFunction(element);
    const cancelled = !manager.finished && index + 1 >= maxPasses;

    if (manager.finished || cancelled) {
      const duration = Date.now() - start;
      await manager.afterEachPass({
        index,
        finished: true,
        cancelled,
        renderDuration: duration,
        resolveDuration: 0
      });

      if (afterEachPass) {
        await afterEachPass({
          index,
          finished: true,
          cancelled,
          renderDuration: duration,
          resolveDuration: 0
        });
      }

      return result;
    } else {
      const resolveStart = Date.now();
      const renderDuration = resolveStart - start;
      await manager.resolve();
      const resolveDuration = Date.now() - resolveStart;
      let performNextPass = true;
      performNextPass = shouldContinue(await manager.afterEachPass({
        index,
        finished: false,
        cancelled: false,
        renderDuration,
        resolveDuration
      })) && performNextPass;

      if (afterEachPass) {
        performNextPass = shouldContinue(await afterEachPass({
          index,
          finished: false,
          cancelled: false,
          renderDuration,
          resolveDuration
        })) && performNextPass;
      }

      if (performNextPass) {
        await manager.betweenEachPass({
          index,
          finished: false,
          cancelled: false,
          renderDuration,
          resolveDuration
        });

        if (betweenEachPass) {
          await betweenEachPass({
            index,
            finished: false,
            cancelled: false,
            renderDuration,
            resolveDuration
          });
        }
      }

      manager.reset();
      return performNextPass ? perform(index + 1) : result;
    }
  }();
}

function shouldContinue(result) {
  return result !== false;
}

function identity(value) {
  return value;
}

export { extract };
