import React, { forwardRef, useRef, useId, useContext, useImperativeHandle } from 'react';
import { MinusMinor, TickSmallMinor } from '@shopify/polaris-icons';
import { classNames } from '../../utilities/css.js';
import { WithinListboxContext } from '../../utilities/listbox/context.js';
import styles from './Checkbox.scss.js';
import { Choice, helpTextID } from '../Choice/Choice.js';
import { errorTextID } from '../InlineError/InlineError.js';
import { useFeatures } from '../../utilities/features/hooks.js';
import { Icon } from '../Icon/Icon.js';

const Checkbox = /*#__PURE__*/forwardRef(function Checkbox({
  ariaControls,
  ariaDescribedBy: ariaDescribedByProp,
  label,
  labelHidden,
  checked = false,
  helpText,
  disabled,
  id: idProp,
  name,
  value,
  error,
  onChange,
  onFocus,
  onBlur,
  labelClassName,
  fill,
  bleed,
  bleedBlockStart,
  bleedBlockEnd,
  bleedInlineStart,
  bleedInlineEnd
}, ref) {
  const inputNode = useRef(null);
  const uniqId = useId();
  const id = idProp ?? uniqId;
  const isWithinListbox = useContext(WithinListboxContext);
  const {
    polarisSummerEditions2023
  } = useFeatures();
  useImperativeHandle(ref, () => ({
    focus: () => {
      if (inputNode.current) {
        inputNode.current.focus();
      }
    }
  }));
  const handleBlur = () => {
    onBlur && onBlur();
  };
  const handleOnClick = () => {
    if (onChange == null || inputNode.current == null || disabled) {
      return;
    }
    onChange(inputNode.current.checked, id);
    inputNode.current.focus();
  };
  const describedBy = [];
  if (error && typeof error !== 'boolean') {
    describedBy.push(errorTextID(id));
  }
  if (helpText) {
    describedBy.push(helpTextID(id));
  }
  if (ariaDescribedByProp) {
    describedBy.push(ariaDescribedByProp);
  }
  const ariaDescribedBy = describedBy.length ? describedBy.join(' ') : undefined;
  const wrapperClassName = classNames(styles.Checkbox, error && styles.error);
  const isIndeterminate = checked === 'indeterminate';
  const isChecked = !isIndeterminate && Boolean(checked);
  const indeterminateAttributes = isIndeterminate ? {
    indeterminate: 'true',
    'aria-checked': 'mixed'
  } : {
    'aria-checked': isChecked
  };
  const iconSource = isIndeterminate ? MinusMinor : TickSmallMinor;
  const animatedTickIcon = polarisSummerEditions2023 && !isIndeterminate;
  const iconSourceSe23 = /*#__PURE__*/React.createElement("svg", {
    viewBox: "0 0 16 16",
    shapeRendering: "geometricPrecision",
    textRendering: "geometricPrecision"
  }, /*#__PURE__*/React.createElement("path", {
    className: classNames(checked && styles.checked),
    d: "M1.5,5.5L3.44655,8.22517C3.72862,8.62007,4.30578,8.64717,4.62362,8.28044L10.5,1.5",
    transform: "translate(2 2.980376)",
    opacity: "0",
    fill: "none",
    stroke: "#fff",
    strokeWidth: "2",
    strokeLinecap: "round",
    strokeLinejoin: "round",
    pathLength: "1"
  }));
  const inputClassName = classNames(styles.Input, isIndeterminate && styles['Input-indeterminate']);
  const extraChoiceProps = {
    helpText,
    error,
    bleed,
    bleedBlockStart,
    bleedBlockEnd,
    bleedInlineStart,
    bleedInlineEnd
  };
  return /*#__PURE__*/React.createElement(Choice, Object.assign({
    id: id,
    label: label,
    labelHidden: labelHidden,
    disabled: disabled,
    labelClassName: classNames(styles.ChoiceLabel, labelClassName),
    fill: fill
  }, extraChoiceProps), /*#__PURE__*/React.createElement("span", {
    className: wrapperClassName
  }, /*#__PURE__*/React.createElement("input", Object.assign({
    ref: inputNode,
    id: id,
    name: name,
    value: value,
    type: "checkbox",
    checked: isChecked,
    disabled: disabled,
    className: inputClassName,
    onBlur: handleBlur,
    onChange: noop,
    onClick: handleOnClick,
    onFocus: onFocus,
    "aria-invalid": error != null,
    "aria-controls": ariaControls,
    "aria-describedby": ariaDescribedBy,
    role: isWithinListbox ? 'presentation' : 'checkbox'
  }, indeterminateAttributes)), /*#__PURE__*/React.createElement("span", {
    className: styles.Backdrop,
    onClick: stopPropagation,
    onKeyUp: stopPropagation
  }), /*#__PURE__*/React.createElement("span", {
    className: classNames(styles.Icon, animatedTickIcon && styles.animated)
  }, animatedTickIcon ? iconSourceSe23 : /*#__PURE__*/React.createElement(Icon, {
    source: iconSource
  }))));
});
function noop() {}
function stopPropagation(event) {
  event.stopPropagation();
}

export { Checkbox };
