import React, { useState, useMemo, useEffect } from 'react';
import { Autocomplete, Icon } from '@shopify/polaris';
import { ClockMinor } from '@shopify/polaris-icons';
import { useI18n } from '@shopify/react-i18n';
import { getLocalizedTimeForDate, formatDateListAsOptionList, generateTimes, getValidDateForTime } from './utilities.js';
import { getDateTimeInShopTimeZone, getDateInShopTimeZone } from '../../utilities/dates.js';

const TIME_BLOCKLIST_REGEX = /[^\d\s:apmAPM]/g;
function TimePicker({
  label,
  labelHidden = false,
  time,
  disabled = false,
  disableTimesBefore
}) {
  const [i18n] = useI18n();
  const ianaTimezone = i18n.defaultTimezone;
  const locale = i18n.locale;
  const selectedDate = getDateTimeInShopTimeZone(time.value, ianaTimezone);
  const localeFormattedTime = getLocalizedTimeForDate(selectedDate, locale);
  const [userInput, setUserInput] = useState(localeFormattedTime);
  const disableTimesBeforeInShopTZ = useMemo(() => {
    return disableTimesBefore ? getDateTimeInShopTimeZone(disableTimesBefore, ianaTimezone) : undefined;
  }, [disableTimesBefore, ianaTimezone]);
  const options = useMemo(() => formatDateListAsOptionList(generateTimes(selectedDate, disableTimesBeforeInShopTZ), locale, ianaTimezone), [selectedDate, disableTimesBeforeInShopTZ, locale, ianaTimezone]);
  useEffect(() => {
    setUserInput(localeFormattedTime);
  }, [localeFormattedTime]);
  const handleTextFieldChange = inputValue => setUserInput(inputValue.replace(TIME_BLOCKLIST_REGEX, ''));
  const handleTextFieldBlur = () => {
    var _time$onBlur;
    if (userInput === localeFormattedTime) {
      return;
    }
    const requestedNewTime = getValidDateForTime(userInput, selectedDate, ianaTimezone);
    const userInputInShopTimeZone = requestedNewTime && getDateInShopTimeZone(requestedNewTime, ianaTimezone);
    const hasDisabledTimesAndInputIsAfterDisabledTime = userInputInShopTimeZone && disableTimesBeforeInShopTZ && userInputInShopTimeZone >= disableTimesBeforeInShopTZ;

    // set the time if the user input is a valid time string and the browser time in the shop TZ is not before disableTimesBefore
    if (requestedNewTime && (!disableTimesBeforeInShopTZ || hasDisabledTimesAndInputIsAfterDisabledTime)) {
      time.onChange(requestedNewTime.toISOString());
    } else {
      setUserInput(localeFormattedTime);
    }
    (_time$onBlur = time.onBlur) === null || _time$onBlur === void 0 ? void 0 : _time$onBlur.call(time);
  };
  return /*#__PURE__*/React.createElement(Autocomplete, {
    options: options,
    selected: [time.value],
    onSelect: selected => time.onChange(selected[0]),
    textField: /*#__PURE__*/React.createElement(Autocomplete.TextField, {
      label: label,
      labelHidden: labelHidden,
      prefix: /*#__PURE__*/React.createElement(Icon, {
        source: ClockMinor,
        color: "subdued"
      }),
      placeholder: i18n.translate('DiscountAppComponents.TimePicker.timePlaceholder'),
      autoComplete: "off",
      error: time.error,
      onBlur: handleTextFieldBlur,
      onChange: handleTextFieldChange,
      value: userInput,
      disabled: disabled
    })
  });
}

export { TimePicker };
