import React from 'react';
import { List } from '@shopify/polaris';
import { useI18n } from '@shopify/react-i18n';
import { RequirementType } from '../../../../constants.js';

const I18N_SCOPE = {
  scope: 'DiscountAppComponents.SummaryCard.MinimumRequirements'
};
function MinimumRequirements({
  requirementType,
  quantity,
  subtotal,
  currencyCode
}) {
  const [i18n] = useI18n();
  switch (requirementType) {
    case RequirementType.None:
      return /*#__PURE__*/React.createElement(List.Item, null, i18n.translate('noMinimumPurchaseRequirement', I18N_SCOPE));
    case RequirementType.Subtotal:
      // eslint-disable-next-line no-case-declarations
      const validSubtotal = getValidSubtotal(i18n, currencyCode, subtotal);
      return validSubtotal ? /*#__PURE__*/React.createElement(List.Item, null, i18n.translate('minimumPurchaseSubtotal', I18N_SCOPE, {
        subtotal: validSubtotal
      })) : null;
    case RequirementType.Quantity:
      return isValidQuantity(quantity) ? /*#__PURE__*/React.createElement(List.Item, null, i18n.translate('minimumPurchaseQuantity', I18N_SCOPE, {
        count: Number(quantity)
      })) : null;
    default:
      return null;
  }
}
function getValidSubtotal(i18n, currencyCode, subtotal) {
  if (subtotal === undefined || currencyCode === undefined) {
    return null;
  }
  const subtotalNumber = Number(subtotal);
  const validatedSubtotal = isNaN(subtotalNumber) ? '' : i18n.formatCurrency(Math.abs(subtotalNumber), {
    currency: currencyCode
  });
  if (subtotal.length > 0 && subtotalNumber > 0 && validatedSubtotal.length > 0) {
    return validatedSubtotal;
  }
  return null;
}
function isValidQuantity(quantity) {
  if (quantity === undefined) {
    return false;
  }
  const quantityNumber = Number(quantity);
  const validatedQuantity = isNaN(quantityNumber) ? '' : Math.abs(quantityNumber).toString();
  return quantity.length > 0 && quantityNumber > 0 && validatedQuantity.length > 0;
}

export { MinimumRequirements };
