import React, { useEffect } from 'react';
import { Text, LegacyStack, InlineError, TextField, LegacyCard, ChoiceList } from '@shopify/polaris';
import { useI18n } from '@shopify/react-i18n';
import styles from './MinimumRequirementsCard.scss.js';
import { forcePositiveInteger } from '../../utilities/numbers.js';
import { RequirementType, DiscountMethod, AppliesTo } from '../../constants.js';
import CurrencyField from '../CurrencyField/CurrencyField.js';

const SUBTOTAL_FIELD_ID = 'discountSubtotal';
const QUANTITY_FIELD_ID = 'discountQuantity';
const AUTOMATIC_REQUIREMENT_TYPES = [RequirementType.Subtotal, RequirementType.Quantity];

// see https://github.com/Shopify/web/pull/12361
const SUBTOTAL_MAX_LENGTH = 15;

// see https://github.com/Shopify/web/pull/11506
const QUANTITY_MAX_LENGTH = 10;
function MinimumRequirementsCard({
  appliesTo,
  currencyCode,
  requirementType,
  subtotal,
  quantity,
  discountMethod,
  isRecurring = false
}) {
  const [i18n] = useI18n();

  // RequirementType.None is not valid for an automatic discount, so if the discount method changes we adjust the requirementType if the new state is invalid.
  useEffect(() => {
    if (discountMethod === DiscountMethod.Automatic && requirementType.value === RequirementType.None) {
      requirementType.onChange(RequirementType.Subtotal);
    }
  }, [discountMethod, requirementType]);
  const currencySymbol = i18n.getCurrencySymbol(currencyCode).symbol;
  const minimumPurchaseLabel = i18n.translate('DiscountAppComponents.MinimumRequirementsCard.minimumSubtotal', {
    currencySymbol: currencySymbol.trim()
  });
  const minimumQuantityLabel = i18n.translate('DiscountAppComponents.MinimumRequirementsCard.minimumQuantity');
  const fieldHelpTextMarkup = /*#__PURE__*/React.createElement(Text, {
    as: "span",
    color: "subdued"
  }, getFieldHelpText(isRecurring, appliesTo, i18n));
  const allMinimumRequirementChoices = [{
    label: i18n.translate('DiscountAppComponents.MinimumRequirementsCard.none'),
    value: RequirementType.None
  }, {
    label: minimumPurchaseLabel,
    value: RequirementType.Subtotal,
    renderChildren: isSelected => {
      return isSelected && /*#__PURE__*/React.createElement(LegacyStack, {
        vertical: true,
        spacing: "extraTight"
      }, /*#__PURE__*/React.createElement("div", {
        className: styles.TextField
      }, /*#__PURE__*/React.createElement(CurrencyField, {
        id: SUBTOTAL_FIELD_ID,
        label: minimumPurchaseLabel,
        labelHidden: true,
        value: subtotal.value,
        maxLength: SUBTOTAL_MAX_LENGTH,
        error: Boolean(subtotal.error),
        currencyCode: currencyCode,
        onChange: subtotal.onChange,
        onBlur: subtotal.onBlur,
        positiveOnly: true
      })), fieldHelpTextMarkup, subtotal.error && /*#__PURE__*/React.createElement(InlineError, {
        fieldID: SUBTOTAL_FIELD_ID,
        message: subtotal.error
      }));
    }
  }, {
    label: minimumQuantityLabel,
    value: RequirementType.Quantity,
    renderChildren: isSelected => {
      return isSelected && /*#__PURE__*/React.createElement(LegacyStack, {
        vertical: true,
        spacing: "extraTight"
      }, /*#__PURE__*/React.createElement("div", {
        className: styles.TextField
      }, /*#__PURE__*/React.createElement(TextField, {
        id: QUANTITY_FIELD_ID,
        label: minimumQuantityLabel,
        labelHidden: true,
        autoComplete: "off",
        error: Boolean(quantity.error),
        value: quantity.value,
        maxLength: QUANTITY_MAX_LENGTH,
        onChange: nextValue => quantity.onChange(forcePositiveInteger(nextValue)),
        onBlur: quantity.onBlur
      })), fieldHelpTextMarkup, quantity.error && /*#__PURE__*/React.createElement(InlineError, {
        fieldID: QUANTITY_FIELD_ID,
        message: quantity.error
      }));
    }
  }];
  const minimumRequirementChoicesToRender = discountMethod === DiscountMethod.Automatic ? allMinimumRequirementChoices.filter(({
    value
  }) => AUTOMATIC_REQUIREMENT_TYPES.includes(value)) : allMinimumRequirementChoices;
  return /*#__PURE__*/React.createElement(LegacyCard, {
    title: i18n.translate('DiscountAppComponents.MinimumRequirementsCard.title'),
    sectioned: true
  }, /*#__PURE__*/React.createElement(ChoiceList, {
    title: i18n.translate('DiscountAppComponents.MinimumRequirementsCard.title'),
    titleHidden: true,
    choices: minimumRequirementChoicesToRender,
    selected: [requirementType.value],
    onChange: nextValue => requirementType.onChange(nextValue[0])
  }));
}
function getFieldHelpText(isRecurring, appliesTo, i18n) {
  const scope = isRecurring ? 'DiscountAppComponents.MinimumRequirementsCard.subscriptions' : 'DiscountAppComponents.MinimumRequirementsCard.oneTime';
  switch (appliesTo) {
    case AppliesTo.Order:
      return i18n.translate('appliesToAllProducts', {
        scope
      });
    case AppliesTo.Products:
      return i18n.translate('appliesToProducts', {
        scope
      });
    case AppliesTo.Collections:
      return i18n.translate('appliesToCollections', {
        scope
      });
  }
}

export { MinimumRequirementsCard };
