import React, { useState, useMemo, useEffect } from 'react';
import { Popover, TextField, Icon, DatePicker as DatePicker$1 } from '@shopify/polaris';
import { CalendarMajor } from '@shopify/polaris-icons';
import { timeFormat, timeParse } from 'd3-time-format';
import { useI18n } from '@shopify/react-i18n';
import { DEFAULT_WEEK_START_DAY } from '../../constants.js';
import { getDateTimeInShopTimeZone, getNewDateAtStartOfDay, getDateInUTC } from '../../utilities/dates.js';

const DATE_BLOCKLIST_REGEX = /[^\d-]/g;
const VALID_DATE_REGEX = /^\d{4}-\d{1,2}-\d{1,2}$/;
function DatePicker({
  date,
  label,
  labelHidden = false,
  disabled = false,
  disableDatesBefore,
  weekStartsOn = DEFAULT_WEEK_START_DAY
}) {
  const [i18n] = useI18n();
  const ianaTimezone = i18n.defaultTimezone;
  const selectedDate = getDateTimeInShopTimeZone(date.value, ianaTimezone);
  const localeFormattedDate = getFormattedDate(selectedDate);
  const [datePickerView, setDatePickerView] = useState({
    month: selectedDate.getMonth(),
    year: selectedDate.getFullYear()
  });
  const [userInput, setUserInput] = useState(localeFormattedDate);
  const [error, setError] = useState();
  const [isActive, setIsActive] = useState(false);
  const disableDatesBeforeInShopTZAtStartOfDay = useMemo(() => {
    return disableDatesBefore ? getNewDateAtStartOfDay(getDateTimeInShopTimeZone(disableDatesBefore, ianaTimezone)) : undefined;
  }, [disableDatesBefore, ianaTimezone]);
  useEffect(() => {
    const selectedDate = getDateTimeInShopTimeZone(date.value, ianaTimezone);
    setDatePickerView({
      month: selectedDate.getMonth(),
      year: selectedDate.getFullYear()
    });
    setUserInput(getFormattedDate(selectedDate));
  }, [date.value, ianaTimezone]);
  const handleMonthChange = (month, year) => {
    setDatePickerView({
      month,
      year
    });
  };
  const handleTextFieldChange = inputValue => setUserInput(inputValue.replace(DATE_BLOCKLIST_REGEX, ''));
  const handleTextFieldBlur = () => {
    var _date$onBlur;
    if (userInput === localeFormattedDate) {
      return;
    }
    setError(null);
    if (userInput === '') {
      setUserInput(localeFormattedDate);
    } else if (isValidDateString(userInput)) {
      const newDate = timeParse('%Y-%m-%d')(userInput);
      const hasDisabledDatesAndInputIsAfterDisabledDate = disableDatesBeforeInShopTZAtStartOfDay && newDate >= disableDatesBeforeInShopTZAtStartOfDay;
      if (!disableDatesBeforeInShopTZAtStartOfDay || hasDisabledDatesAndInputIsAfterDisabledDate) {
        setDatePickerView({
          month: newDate.getMonth(),
          year: newDate.getFullYear()
        });
        handleDateChange(getDateTimeInShopTimeZone(date.value, ianaTimezone), newDate, date.onChange, ianaTimezone);
      } else {
        setUserInput(localeFormattedDate);
      }
    } else {
      setError(i18n.translate('DiscountAppComponents.DatePicker.dateFormatError'));
    }
    (_date$onBlur = date.onBlur) === null || _date$onBlur === void 0 ? void 0 : _date$onBlur.call(date);
  };
  const handleDatePickerChange = value => {
    const formattedDate = getFormattedDate(value.start);
    setError(null);
    setUserInput(formattedDate);
    setIsActive(false);
    setDatePickerView({
      month: value.start.getMonth(),
      year: value.start.getFullYear()
    });
    handleDateChange(getDateTimeInShopTimeZone(date.value, ianaTimezone), value.start, date.onChange, ianaTimezone);
  };
  return /*#__PURE__*/React.createElement(Popover, {
    activator: /*#__PURE__*/React.createElement(TextField, {
      value: userInput,
      label: label,
      labelHidden: labelHidden,
      prefix: /*#__PURE__*/React.createElement(Icon, {
        source: CalendarMajor,
        color: "subdued"
      }),
      placeholder: i18n.translate('DiscountAppComponents.DatePicker.datePlaceholder'),
      error: date.error || error || false,
      autoComplete: "off",
      onFocus: () => setIsActive(true),
      onChange: handleTextFieldChange,
      onBlur: handleTextFieldBlur,
      disabled: disabled
    }),
    active: isActive,
    onClose: () => setIsActive(false),
    autofocusTarget: "none",
    sectioned: true
  }, /*#__PURE__*/React.createElement(DatePicker$1, {
    month: datePickerView.month,
    year: datePickerView.year,
    disableDatesBefore: disableDatesBeforeInShopTZAtStartOfDay,
    weekStartsOn: weekStartsOn,
    onMonthChange: handleMonthChange,
    selected: selectedDate,
    onChange: handleDatePickerChange
  }));
}
function isValidDateString(date) {
  if (!date) {
    return false;
  }
  return VALID_DATE_REGEX.test(date) && !isNaN(new Date(date).getTime());
}
const getFormattedDate = selectedDate => timeFormat('%Y-%m-%d')(selectedDate);
const handleDateChange = (oldDate, newDate, onChange, ianaTimezone) => {
  // Re-apply the current time before firing the datetime field onChange.
  newDate.setHours(oldDate.getHours(), oldDate.getMinutes(), oldDate.getSeconds(), oldDate.getMilliseconds());
  if (oldDate.getTime() !== newDate.getTime()) {
    // Persist date in UTC
    onChange(getDateInUTC(newDate, ianaTimezone).toISOString());
  }
};

export { DatePicker };
