import React from 'react';
import { LegacyCard, LegacyStack, Banner, Link, Text, ChoiceList } from '@shopify/polaris';
import { useI18n } from '@shopify/react-i18n';
import { DiscountClass } from '../../constants.js';
import { HelpText } from './components/HelpText/HelpText.js';

const I18N_SCOPE = {
  scope: 'DiscountAppComponents.CombinationCard'
};
function CombinationCard({
  discountClass,
  discountDescriptor,
  combinableDiscountTypes,
  combinableDiscountCounts,
  discountId
}) {
  const [i18n] = useI18n();
  const handleDiscountCombinesWithChange = selectedChoices => {
    combinableDiscountTypes.onChange({
      orderDiscounts: selectedChoices.includes(DiscountClass.Order),
      productDiscounts: selectedChoices.includes(DiscountClass.Product),
      shippingDiscounts: selectedChoices.includes(DiscountClass.Shipping)
    });
  };
  const trimmedDescriptor = discountDescriptor.trim();
  const selectedChoices = getSelectedChoices(combinableDiscountTypes.value);
  const shouldShowBanner = discountClass === DiscountClass.Order && (selectedChoices.includes(DiscountClass.Product) || selectedChoices.includes(DiscountClass.Order)) || discountClass === DiscountClass.Product && selectedChoices.includes(DiscountClass.Order);
  return /*#__PURE__*/React.createElement(LegacyCard, {
    title: i18n.translate('title', I18N_SCOPE),
    sectioned: true
  }, /*#__PURE__*/React.createElement(LegacyStack, {
    vertical: true,
    spacing: "baseTight"
  }, shouldShowBanner && /*#__PURE__*/React.createElement(Banner, {
    title: i18n.translate('warning.title', I18N_SCOPE),
    status: "warning"
  }, /*#__PURE__*/React.createElement("p", null, i18n.translate('warning.description', I18N_SCOPE), ' ', /*#__PURE__*/React.createElement(Link, {
    url: `https://help.shopify.com/${i18n.locale}/manual/discounts/combining-discounts`,
    external: true
  }, i18n.translate('warning.link', I18N_SCOPE)))), /*#__PURE__*/React.createElement("p", null, trimmedDescriptor ? /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(Text, {
    as: "span",
    fontWeight: "semibold"
  }, trimmedDescriptor), ' ', i18n.translate('discountNameFilled', I18N_SCOPE)) : i18n.translate('discountNameNotFilled', I18N_SCOPE, {
    discountClass: i18n.translate(`discountClass.${discountClass.toLowerCase()}`, I18N_SCOPE)
  })), /*#__PURE__*/React.createElement(ChoiceList, {
    title: i18n.translate('combinesWith', I18N_SCOPE),
    titleHidden: true,
    allowMultiple: true,
    choices: buildChoices({
      discountClass,
      discountId,
      discountDescriptor,
      i18n,
      combinableDiscountCounts
    }),
    selected: getSelectedChoices(combinableDiscountTypes.value),
    onChange: handleDiscountCombinesWithChange
  })));
}
function buildChoices({
  discountClass: currentDiscountClass,
  combinableDiscountCounts,
  discountId: currentDiscountId,
  discountDescriptor,
  i18n
}) {
  const hasCounts = typeof combinableDiscountCounts !== 'undefined';
  const productOption = {
    label: i18n.translate('options.productLabel', I18N_SCOPE),
    value: DiscountClass.Product,
    renderChildren: isSelected => isSelected && hasCounts ? /*#__PURE__*/React.createElement(HelpText, {
      currentDiscountClass: currentDiscountClass,
      targetDiscountClass: DiscountClass.Product,
      count: getActualCombiningDiscountsCount(combinableDiscountCounts.productDiscountsCount, currentDiscountClass === DiscountClass.Product, currentDiscountId),
      currentDiscountId: currentDiscountId,
      currentDiscountName: discountDescriptor
    }) : null
  };
  const orderOption = {
    label: i18n.translate('options.orderLabel', I18N_SCOPE),
    value: DiscountClass.Order,
    renderChildren: isSelected => isSelected && hasCounts ? /*#__PURE__*/React.createElement(HelpText, {
      currentDiscountClass: currentDiscountClass,
      targetDiscountClass: DiscountClass.Order,
      count: getActualCombiningDiscountsCount(combinableDiscountCounts.orderDiscountsCount, currentDiscountClass === DiscountClass.Order, currentDiscountId),
      currentDiscountId: currentDiscountId,
      currentDiscountName: discountDescriptor
    }) : null
  };
  const shippingOption = {
    label: i18n.translate('options.shippingLabel', I18N_SCOPE),
    value: DiscountClass.Shipping,
    renderChildren: isSelected => isSelected && hasCounts ? /*#__PURE__*/React.createElement(HelpText, {
      currentDiscountClass: currentDiscountClass,
      targetDiscountClass: DiscountClass.Shipping,
      count: getActualCombiningDiscountsCount(combinableDiscountCounts.shippingDiscountsCount, currentDiscountClass === DiscountClass.Shipping, currentDiscountId),
      currentDiscountId: currentDiscountId,
      currentDiscountName: discountDescriptor
    }) : null
  };
  switch (currentDiscountClass) {
    case DiscountClass.Product:
      return [productOption, orderOption, shippingOption];
    case DiscountClass.Order:
      return [productOption, orderOption, shippingOption];
    case DiscountClass.Shipping:
      return [productOption, orderOption];
    default:
      return [{
        label: '',
        value: ''
      }];
  }
}

/**
 * The combines with count needs to exclude the current discount if:
 *  - the current discount is of the same type as the combination discount type
 *  - the current discount is saved
 */
function getActualCombiningDiscountsCount(numCombinableDiscountsForClass, discountClassesMatch, currentDiscountId) {
  if (discountClassesMatch && Boolean(currentDiscountId)) {
    return numCombinableDiscountsForClass - 1;
  }
  return numCombinableDiscountsForClass;
}
const getSelectedChoices = combinableDiscountTypes => [...(combinableDiscountTypes.productDiscounts ? [DiscountClass.Product] : []), ...(combinableDiscountTypes.orderDiscounts ? [DiscountClass.Order] : []), ...(combinableDiscountTypes.shippingDiscounts ? [DiscountClass.Shipping] : [])];

export { CombinationCard };
