'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var functionEnhancers = require('@shopify/function-enhancers');
var sanitiseDateString = require('./sanitise-date-string.js');
var formatDate = require('./utilities/formatDate.js');

const TWO_DIGIT_REGEX = /(\d{2})/;
function getDateTimeParts(date, timeZone) {
  return {
    year: () => DateTimeParts.getYear(date, timeZone),
    month: () => DateTimeParts.getMonth(date, timeZone),
    day: () => DateTimeParts.getDay(date, timeZone),
    weekday: () => DateTimeParts.getWeekday(date, timeZone),
    hour: () => DateTimeParts.getHour(date, timeZone),
    minute: () => DateTimeParts.getMinute(date, timeZone),
    second: () => DateTimeParts.getSecond(date, timeZone)
  };
}

function dateTimeCacheKey(unit) {
  return (date, timeZone) => `${unit}-${date.toString()}-${timeZone}`;
}

var Weekday;

(function (Weekday) {
  Weekday["Monday"] = "Monday";
  Weekday["Tuesday"] = "Tuesday";
  Weekday["Wednesday"] = "Wednesday";
  Weekday["Thursday"] = "Thursday";
  Weekday["Friday"] = "Friday";
  Weekday["Saturday"] = "Saturday";
  Weekday["Sunday"] = "Sunday";
})(Weekday || (Weekday = {}));

const weekdays = {
  Monday: 0,
  Tuesday: 1,
  Wednesday: 2,
  Thursday: 3,
  Friday: 4,
  Saturday: 5,
  Sunday: 6
};

function isWeekday(weekday) {
  return Object.keys(weekdays).some(key => key === weekday);
}

function assertNever(message) {
  throw new Error(message);
}

function getWeekdayValue(weekday) {
  if (!isWeekday(weekday)) {
    return assertNever(`Unexpected weekday: ${weekday}`);
  }

  return weekdays[weekday];
} // eslint-disable-next-line @typescript-eslint/no-extraneous-class


class DateTimeParts {}

DateTimeParts.getYear = functionEnhancers.memoize((date, timeZone) => {
  if (isNaN(date.valueOf())) {
    throw new Error(`Unable to parse date: ${date} for timezone: ${timeZone}`);
  }

  const yearString = formatDate.formatDate(date, 'en', {
    timeZone,
    year: 'numeric'
  });
  const sanitisedYearString = sanitiseDateString.sanitiseDateString(yearString);
  const year = parseInt(sanitisedYearString, 10);

  if (isNaN(year)) {
    throw new Error(`Unable to parse year: '${yearString}'`);
  }

  return year;
}, dateTimeCacheKey('year'));
DateTimeParts.getMonth = functionEnhancers.memoize((date, timeZone) => {
  const monthString = formatDate.formatDate(date, 'en', {
    timeZone,
    month: 'numeric'
  });
  const sanitisedMonthString = sanitiseDateString.sanitiseDateString(monthString);
  const month = parseInt(sanitisedMonthString, 10);

  if (isNaN(month)) {
    throw new Error(`Unable to parse month: '${monthString}'`);
  }

  return month;
}, dateTimeCacheKey('month'));
DateTimeParts.getDay = functionEnhancers.memoize((date, timeZone) => {
  const dayString = formatDate.formatDate(date, 'en', {
    timeZone,
    day: 'numeric'
  });
  const sanitisedDayString = sanitiseDateString.sanitiseDateString(dayString);
  const day = parseInt(sanitisedDayString, 10);

  if (isNaN(day)) {
    throw new Error(`Unable to parse day: '${dayString}'`);
  }

  return day;
}, dateTimeCacheKey('day'));
DateTimeParts.getWeekday = functionEnhancers.memoize((date, timeZone) => {
  const weekdayString = formatDate.formatDate(date, 'en', {
    timeZone,
    weekday: 'long'
  });
  const sanitisedWeekdayString = sanitiseDateString.sanitiseDateString(weekdayString);
  return getWeekdayValue(sanitisedWeekdayString);
}, dateTimeCacheKey('weekday'));
DateTimeParts.getHour = functionEnhancers.memoize((date, timeZone) => {
  const hourString = formatDate.formatDate(date, 'en', {
    timeZone,
    hour12: false,
    hour: 'numeric'
  });
  let hour = parseInt(hourString, 10);

  if (isNaN(hour)) {
    hour = DateTimeParts.getTimePartsFallback(date, timeZone).hour;
  }

  return hour;
}, dateTimeCacheKey('hour'));
DateTimeParts.getMinute = functionEnhancers.memoize((date, timeZone) => {
  const minuteString = formatDate.formatDate(date, 'en', {
    timeZone,
    minute: 'numeric'
  });
  let minute = parseInt(minuteString, 10);

  if (isNaN(minute)) {
    minute = DateTimeParts.getTimePartsFallback(date, timeZone).minute;
  }

  return minute;
}, dateTimeCacheKey('minute'));
DateTimeParts.getSecond = functionEnhancers.memoize((date, timeZone) => {
  const secondString = formatDate.formatDate(date, 'en', {
    timeZone,
    second: 'numeric'
  });
  let second = parseInt(secondString, 10);

  if (isNaN(second)) {
    second = DateTimeParts.getTimePartsFallback(date, timeZone).second;
  }

  return second;
}, dateTimeCacheKey('second'));
DateTimeParts.getTimePartsFallback = functionEnhancers.memoize((date, timeZone) => {
  const timeString = formatDate.formatDate(date, 'en', {
    timeZone,
    hour12: false,
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  }); // In Microsoft Edge, Intl.DateTimeFormat returns invisible characters around the individual numbers

  const [dirtyHour, dirtyMinute, dirtySecond] = timeString.split(':');
  const rawHour = new RegExp(TWO_DIGIT_REGEX).exec(dirtyHour);
  const rawMinute = new RegExp(TWO_DIGIT_REGEX).exec(dirtyMinute);
  const rawSecond = new RegExp(TWO_DIGIT_REGEX).exec(dirtySecond);

  if (rawHour != null && rawMinute != null && rawSecond != null) {
    const hour = parseInt(rawHour[1], 10);
    const minute = parseInt(rawMinute[1], 10);
    const second = parseInt(rawSecond[1], 10);
    return {
      hour,
      minute,
      second
    };
  }

  throw new Error(`Unable to parse timeString: '${timeString}'`);
}, dateTimeCacheKey('timePartsFallback'));

exports.getDateTimeParts = getDateTimeParts;
