import { getExtensionIds } from '../context/id-matching.js';
import { MigrateFlowExtensionMutation, } from '../../api/graphql/extension_migrate_flow_extension.js';
import { partnersRequest } from '@shopify/cli-kit/node/api/partners';
import { ensureAuthenticatedPartners } from '@shopify/cli-kit/node/session';
import { AbortError } from '@shopify/cli-kit/node/error';
import { slugify } from '@shopify/cli-kit/common/string';
export function getFlowExtensionsToMigrate(localSources, remoteSources, identifiers) {
    const ids = getExtensionIds(localSources, identifiers);
    const localExtensionTypesToMigrate = ['flow_action', 'flow_trigger'];
    const remoteExtensionTypesToMigrate = ['flow_action_definition', 'flow_trigger_definition'];
    const typesMap = new Map([
        ['flow_action', 'flow_action_definition'],
        ['flow_trigger', 'flow_trigger_definition'],
    ]);
    const local = localSources.filter((source) => localExtensionTypesToMigrate.includes(source.type));
    const remote = remoteSources.filter((source) => remoteExtensionTypesToMigrate.includes(source.type));
    // Map remote sources by uuid and slugified title (the slugified title is used for matching with local folder)
    const remoteSourcesMap = new Map();
    remote.forEach((remoteSource) => {
        remoteSourcesMap.set(remoteSource.uuid, remoteSource);
        remoteSourcesMap.set(slugify(remoteSource.title), remoteSource);
    });
    return local.reduce((accumulator, localSource) => {
        const localSourceId = ids[localSource.localIdentifier] ?? 'unknown';
        const remoteSource = remoteSourcesMap.get(localSourceId) || remoteSourcesMap.get(localSource.localIdentifier);
        const typeMatch = typesMap.get(localSource.type) === remoteSource?.type;
        if (remoteSource && typeMatch) {
            accumulator.push({ local: localSource, remote: remoteSource });
        }
        return accumulator;
    }, []);
}
export async function migrateFlowExtensions(extensionsToMigrate, appId, remoteExtensions) {
    const migratedIDs = await Promise.all(extensionsToMigrate.map(({ remote }) => migrateFlowExtension(appId, remote.id)));
    const typesMap = new Map([
        ['flow_action_definition', 'FLOW_ACTION'],
        ['flow_trigger_definition', 'FLOW_TRIGGER'],
    ]);
    return remoteExtensions
        .filter((extension) => migratedIDs.includes(extension.id))
        .map((extension) => {
        return {
            ...extension,
            type: typesMap.get(extension.type) ?? extension.type,
        };
    });
}
export async function migrateFlowExtension(apiKey, registrationId) {
    const token = await ensureAuthenticatedPartners();
    const variables = {
        apiKey,
        registrationId,
    };
    const result = await partnersRequest(MigrateFlowExtensionMutation, token, variables);
    if (result?.migrateFlowExtension?.userErrors?.length > 0) {
        const errors = result.migrateFlowExtension.userErrors.map((error) => error.message).join(', ');
        throw new AbortError(errors);
    }
    if (!result?.migrateFlowExtension?.migratedFlowExtension) {
        throw new AbortError("Couldn't migrate to Flow extension");
    }
    return registrationId;
}
//# sourceMappingURL=migrate-flow-extension.js.map