import { manualMatchIds } from './id-manual-matching.js';
import { automaticMatchmaking } from './id-matching.js';
import { deployConfirmationPrompt, extensionMigrationPrompt, matchConfirmationPrompt } from './prompts.js';
import { createExtension } from '../dev/create-extension.js';
import { getUIExtensionsToMigrate, migrateExtensionsToUIExtension } from '../dev/migrate-to-ui-extension.js';
import { getFlowExtensionsToMigrate, migrateFlowExtensions } from '../dev/migrate-flow-extension.js';
import { err, ok } from '@shopify/cli-kit/node/result';
import { ensureAuthenticatedPartners } from '@shopify/cli-kit/node/session';
import { outputCompleted } from '@shopify/cli-kit/node/output';
export async function ensureExtensionsIds(options, { extensionRegistrations: initialRemoteExtensions, dashboardManagedExtensionRegistrations: dashboardOnlyExtensions, }) {
    let remoteExtensions = initialRemoteExtensions;
    const validIdentifiers = options.envIdentifiers.extensions ?? {};
    let localExtensions = options.app.allExtensions.filter((ext) => !ext.isFunctionExtension);
    if (options.deploymentMode === 'unified' || options.deploymentMode === 'unified-skip-release') {
        const functionExtensions = options.app.allExtensions.filter((ext) => ext.isFunctionExtension);
        functionExtensions.forEach((ext) => (ext.usingExtensionsFramework = true));
        localExtensions = localExtensions.concat(functionExtensions);
    }
    const uiExtensionsToMigrate = getUIExtensionsToMigrate(localExtensions, remoteExtensions, validIdentifiers);
    const flowExtensionsToMigrate = getFlowExtensionsToMigrate(localExtensions, dashboardOnlyExtensions, validIdentifiers);
    if (uiExtensionsToMigrate.length > 0) {
        const confirmedMigration = await extensionMigrationPrompt(uiExtensionsToMigrate);
        if (!confirmedMigration)
            return err('user-cancelled');
        remoteExtensions = await migrateExtensionsToUIExtension(uiExtensionsToMigrate, options.appId, remoteExtensions);
    }
    if (flowExtensionsToMigrate.length > 0) {
        const confirmedMigration = await extensionMigrationPrompt(flowExtensionsToMigrate, false);
        if (!confirmedMigration)
            return err('user-cancelled');
        const newRemoteExtensions = await migrateFlowExtensions(flowExtensionsToMigrate, options.appId, dashboardOnlyExtensions);
        remoteExtensions = remoteExtensions.concat(newRemoteExtensions);
    }
    const matchExtensions = await automaticMatchmaking(localExtensions, remoteExtensions, validIdentifiers, 'uuid');
    let validMatches = matchExtensions.identifiers;
    const validMatchesById = {};
    const extensionsToCreate = matchExtensions.toCreate ?? [];
    for (const pending of matchExtensions.toConfirm) {
        // eslint-disable-next-line no-await-in-loop
        const confirmed = await matchConfirmationPrompt(pending.local, pending.remote);
        if (confirmed) {
            validMatches[pending.local.localIdentifier] = pending.remote.uuid;
        }
        else {
            extensionsToCreate.push(pending.local);
        }
    }
    let onlyRemoteExtensions = matchExtensions.toManualMatch.remote ?? [];
    if (matchExtensions.toManualMatch.local.length > 0) {
        const matchResult = await manualMatchIds(matchExtensions.toManualMatch, 'uuid');
        validMatches = { ...validMatches, ...matchResult.identifiers };
        extensionsToCreate.push(...matchResult.toCreate);
        onlyRemoteExtensions = matchResult.onlyRemote;
    }
    if (!options.force) {
        const question = (() => {
            switch (options.deploymentMode) {
                case 'legacy':
                    return 'Make the following changes to your extensions in Shopify Partners?';
                case 'unified':
                    return `Release a new version of ${options.partnersApp?.title}?`;
                case 'unified-skip-release':
                    return `Create a new version of ${options.partnersApp?.title}?`;
            }
        })();
        const confirmed = await deployConfirmationPrompt({
            appTitle: options.partnersApp?.title,
            question,
            identifiers: validMatches,
            toCreate: extensionsToCreate,
            onlyRemote: onlyRemoteExtensions,
            dashboardOnly: options.deploymentMode === 'unified' || options.deploymentMode === 'unified-skip-release'
                ? dashboardOnlyExtensions
                : [],
        }, options.deploymentMode, options.appId, options.token);
        if (!confirmed)
            return err('user-cancelled');
    }
    if (extensionsToCreate.length > 0) {
        const newIdentifiers = await createExtensions(extensionsToCreate, options.appId);
        for (const [localIdentifier, registration] of Object.entries(newIdentifiers)) {
            validMatches[localIdentifier] = registration.uuid;
            validMatchesById[localIdentifier] = registration.id;
        }
    }
    // For extensions we also need the match by ID, not only UUID (doesn't apply to functions)
    for (const [localIdentifier, uuid] of Object.entries(validMatches)) {
        const registration = remoteExtensions.find((registration) => registration.uuid === uuid);
        if (registration)
            validMatchesById[localIdentifier] = registration.id;
    }
    return ok({
        extensions: validMatches,
        extensionIds: validMatchesById,
    });
}
async function createExtensions(extensions, appId) {
    const token = await ensureAuthenticatedPartners();
    const result = {};
    for (const extension of extensions) {
        // Create one at a time to avoid API rate limiting issues.
        // eslint-disable-next-line no-await-in-loop
        const registration = await createExtension(appId, extension.graphQLType, extension.handle, token);
        outputCompleted(`Created extension ${extension.handle}.`);
        result[extension.localIdentifier] = registration;
    }
    return result;
}
//# sourceMappingURL=identifiers-extensions.js.map