import { buildFunctionExtension, buildThemeExtension, buildUIExtension, } from '../../services/build/extension.js';
import { bundleThemeExtension } from '../../services/extensions/bundle.js';
import { uploadWasmBlob } from '../../services/deploy/upload.js';
import { ok } from '@shopify/cli-kit/node/result';
import { constantize, slugify } from '@shopify/cli-kit/common/string';
import { randomUUID } from '@shopify/cli-kit/node/crypto';
import { partnersFqdn } from '@shopify/cli-kit/node/context/fqdn';
import { joinPath } from '@shopify/cli-kit/node/path';
import { useThemebundling } from '@shopify/cli-kit/node/context/local';
import { touchFile, writeFile } from '@shopify/cli-kit/node/fs';
/**
 * Class that represents an instance of a local extension
 * Before creating this class we've validated that:
 * - There is a spec for this type of extension
 * - The Schema for that spec is followed by the extension config toml file
 * - We were able to find an entry point file for that extension
 *
 * It supports extension points, making this Class compatible with both new ui-extension
 * and legacy extension types. Extension points are optional and this class will handle them if present.
 *
 * This class holds the public interface to interact with extensions
 */
export class ExtensionInstance {
    get graphQLType() {
        if (this.features.includes('function')) {
            if (this.useExtensionsFramework)
                return 'FUNCTION';
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            const functionConfig = this.configuration;
            return functionConfig.type.toUpperCase();
        }
        return (this.specification.graphQLType ?? this.specification.identifier).toUpperCase();
    }
    get type() {
        return this.configuration.type;
    }
    get humanName() {
        return this.specification.externalName;
    }
    get name() {
        return this.configuration.name;
    }
    get dependency() {
        return this.specification.dependency;
    }
    get externalType() {
        return this.specification.externalIdentifier;
    }
    get surface() {
        return this.specification.surface;
    }
    get isPreviewable() {
        return this.features.includes('ui_preview');
    }
    get isThemeExtension() {
        return this.features.includes('theme');
    }
    get isFunctionExtension() {
        return this.features.includes('function');
    }
    get isESBuildExtension() {
        return this.features.includes('esbuild');
    }
    get features() {
        return this.specification.appModuleFeatures(this.configuration);
    }
    get outputFileName() {
        return `${this.handle}.js`;
    }
    set usingExtensionsFramework(value) {
        this.useExtensionsFramework = value;
    }
    constructor(options) {
        this.configuration = options.configuration;
        this.configurationPath = options.configurationPath;
        this.entrySourceFilePath = options.entryPath ?? '';
        this.directory = options.directory;
        this.specification = options.specification;
        this.devUUID = `dev-${randomUUID()}`;
        this.handle = this.configuration.handle ?? slugify(this.configuration.name ?? '');
        this.localIdentifier = this.handle;
        this.idEnvironmentVariableName = `SHOPIFY_${constantize(this.localIdentifier)}_ID`;
        this.useExtensionsFramework = false;
        this.outputPath = this.directory;
        if (this.features.includes('esbuild') || this.type === 'tax_calculation') {
            this.outputPath = joinPath(this.directory, 'dist', `${this.outputFileName}`);
        }
        if (this.isFunctionExtension) {
            const config = this.configuration;
            this.outputPath = joinPath(this.directory, config.build.path ?? joinPath('dist', 'index.wasm'));
        }
    }
    isDraftable(unifiedDeployment) {
        if (unifiedDeployment) {
            return !this.isThemeExtension;
        }
        else {
            return !this.isPreviewable && !this.isThemeExtension && !this.isFunctionExtension;
        }
    }
    async deployConfig({ apiKey, token, unifiedDeployment, }) {
        if (this.isFunctionExtension) {
            if (!unifiedDeployment)
                return undefined;
            const { moduleId } = await uploadWasmBlob(this.localIdentifier, this.outputPath, apiKey, token);
            return this.specification.deployConfig?.(this.configuration, this.directory, apiKey, moduleId);
        }
        return (
        // module id param is not necessary for non-Function extensions
        this.specification.deployConfig?.(this.configuration, this.directory, apiKey, undefined) ??
            Promise.resolve(undefined));
    }
    validate() {
        if (!this.specification.validate)
            return Promise.resolve(ok(undefined));
        return this.specification.validate(this.configuration, this.directory);
    }
    preDeployValidation() {
        if (!this.specification.preDeployValidation)
            return Promise.resolve();
        return this.specification.preDeployValidation(this);
    }
    buildValidation() {
        if (!this.specification.buildValidation)
            return Promise.resolve();
        return this.specification.buildValidation(this);
    }
    async publishURL(options) {
        const fqdn = await partnersFqdn();
        const parnersPath = this.specification.partnersWebIdentifier;
        return `https://${fqdn}/${options.orgId}/apps/${options.appId}/extensions/${parnersPath}/${options.extensionId}`;
    }
    // UI Specific properties
    getBundleExtensionStdinContent() {
        if (this.specification.getBundleExtensionStdinContent) {
            return this.specification.getBundleExtensionStdinContent(this.configuration);
        }
        const relativeImportPath = this.entrySourceFilePath?.replace(this.directory, '');
        return `import '.${relativeImportPath}';`;
    }
    shouldFetchCartUrl() {
        return this.specification.shouldFetchCartUrl?.(this.configuration) || false;
    }
    hasExtensionPointTarget(target) {
        return this.specification.hasExtensionPointTarget?.(this.configuration, target) || false;
    }
    // Functions specific properties
    get buildCommand() {
        const config = this.configuration;
        return config.build.command;
    }
    get watchPaths() {
        const config = this.configuration;
        const configuredPaths = config.build.watch ? [config.build.watch].flat() : [];
        if (!this.isJavaScript && configuredPaths.length === 0) {
            return null;
        }
        const watchPaths = configuredPaths ?? [];
        if (this.isJavaScript && configuredPaths.length === 0) {
            watchPaths.push(joinPath('src', '**', '*.js'));
            watchPaths.push(joinPath('src', '**', '*.ts'));
        }
        watchPaths.push(joinPath('**', '!(.)*.graphql'));
        return watchPaths.map((path) => joinPath(this.directory, path));
    }
    get inputQueryPath() {
        return joinPath(this.directory, 'input.graphql');
    }
    get isJavaScript() {
        return Boolean(this.entrySourceFilePath?.endsWith('.js') || this.entrySourceFilePath?.endsWith('.ts'));
    }
    async build(options) {
        if (this.isThemeExtension) {
            return buildThemeExtension(this, options);
        }
        else if (this.isFunctionExtension) {
            return buildFunctionExtension(this, options);
        }
        else if (this.features.includes('esbuild')) {
            return buildUIExtension(this, options);
        }
        // Workaround for tax_calculations because they remote spec NEEDS a valid js file to be included.
        if (this.type === 'tax_calculation') {
            await touchFile(this.outputPath);
            await writeFile(this.outputPath, '(()=>{})();');
        }
    }
    async buildForBundle(options, identifiers, bundleDirectory) {
        const extensionId = identifiers.extensions[this.localIdentifier];
        const outputFile = this.isThemeExtension ? '' : joinPath('dist', `${this.outputFileName}`);
        if (this.features.includes('bundling')) {
            // Modules that are going to be inclued in the bundle should be built in the bundle directory
            this.outputPath = joinPath(bundleDirectory, extensionId, outputFile);
        }
        await this.build(options);
        if (this.isThemeExtension && useThemebundling()) {
            await bundleThemeExtension(this, options);
        }
    }
    async bundleConfig({ identifiers, token, apiKey, unifiedDeployment }) {
        const configValue = await this.deployConfig({ apiKey, token, unifiedDeployment });
        if (!configValue)
            return undefined;
        const { handle, ...remainingConfigs } = configValue;
        const contextValue = handle || '';
        return {
            uuid: identifiers.extensions[this.localIdentifier],
            config: JSON.stringify(remainingConfigs),
            context: contextValue,
        };
    }
}
//# sourceMappingURL=extension-instance.js.map