import { isWebType } from './loader.js';
import { isType } from '../../utilities/types.js';
import { zod } from '@shopify/cli-kit/node/schema';
import { getDependencies, readAndParsePackageJson } from '@shopify/cli-kit/node/node-package-manager';
import { fileRealPath, findPathUp } from '@shopify/cli-kit/node/fs';
import { joinPath } from '@shopify/cli-kit/node/path';
export const LegacyAppSchema = zod
    .object({
    client_id: zod.number().optional(),
    name: zod.string().optional(),
    scopes: zod.string().default(''),
    extension_directories: zod.array(zod.string()).optional(),
    web_directories: zod.array(zod.string()).optional(),
})
    .strict();
// adding http or https presence and absence of new lines to url validation
const validateUrl = (zodType) => {
    return zodType
        .url()
        .refine((value) => Boolean(value.match(/^(https?:\/\/)/)), { message: 'Invalid url' })
        .refine((value) => !value.includes('\n'), { message: 'Invalid url' });
};
export const AppSchema = zod
    .object({
    name: zod.string().max(30),
    client_id: zod.string(),
    application_url: validateUrl(zod.string()),
    embedded: zod.boolean(),
    access_scopes: zod
        .object({
        scopes: zod.string().optional(),
        use_legacy_install_flow: zod.boolean().optional(),
    })
        .optional(),
    auth: zod
        .object({
        redirect_urls: zod.array(validateUrl(zod.string())),
    })
        .optional(),
    webhooks: zod.object({
        api_version: zod.string(),
        privacy_compliance: zod
            .object({
            customer_deletion_url: validateUrl(zod.string()).optional(),
            customer_data_request_url: validateUrl(zod.string()).optional(),
            shop_deletion_url: validateUrl(zod.string()).optional(),
        })
            .optional(),
    }),
    app_proxy: zod
        .object({
        url: validateUrl(zod.string()),
        subpath: zod.string(),
        prefix: zod.string(),
    })
        .optional(),
    pos: zod
        .object({
        embedded: zod.boolean(),
    })
        .optional(),
    app_preferences: zod
        .object({
        url: validateUrl(zod.string().max(255)),
    })
        .optional(),
    build: zod
        .object({
        automatically_update_urls_on_dev: zod.boolean().optional(),
        dev_store_url: zod.string().optional(),
    })
        .optional(),
    extension_directories: zod.array(zod.string()).optional(),
    web_directories: zod.array(zod.string()).optional(),
})
    .strict();
export const AppConfigurationSchema = zod.union([LegacyAppSchema, AppSchema]);
/**
 * Check whether a shopify.app.toml schema is valid against the legacy schema definition.
 * @param item - the item to validate
 */
export function isLegacyAppSchema(item) {
    return isType(LegacyAppSchema, item);
}
/**
 * Check whether a shopify.app.toml schema is valid against the current schema definition.
 * @param item - the item to validate
 */
export function isCurrentAppSchema(item) {
    return isType(AppSchema, item);
}
/**
 * Get scopes from a given app.toml config file.
 * @param config - a configuration file
 */
export function getAppScopes(config) {
    if (isLegacyAppSchema(config)) {
        return config.scopes;
    }
    else {
        return config.access_scopes?.scopes ?? '';
    }
}
/**
 * Get scopes as an array from a given app.toml config file.
 * @param config - a configuration file
 */
export function getAppScopesArray(config) {
    const scopes = getAppScopes(config);
    return scopes.length ? scopes.split(',').map((scope) => scope.trim()) : [];
}
export function usesLegacyScopesBehavior(app) {
    const config = 'configurationPath' in app ? app.configuration : app;
    if (isLegacyAppSchema(config))
        return true;
    return Boolean(config.access_scopes?.use_legacy_install_flow);
}
export function appIsLaunchable(app) {
    const frontendConfig = app?.webs?.find((web) => isWebType(web, WebType.Frontend));
    const backendConfig = app?.webs?.find((web) => isWebType(web, WebType.Backend));
    return Boolean(frontendConfig || backendConfig);
}
export var WebType;
(function (WebType) {
    WebType["Frontend"] = "frontend";
    WebType["Backend"] = "backend";
    WebType["Background"] = "background";
})(WebType || (WebType = {}));
const ensurePathStartsWithSlash = (arg) => (typeof arg === 'string' && !arg.startsWith('/') ? `/${arg}` : arg);
const WebConfigurationAuthCallbackPathSchema = zod.preprocess(ensurePathStartsWithSlash, zod.string());
const baseWebConfigurationSchema = zod.object({
    auth_callback_path: zod
        .union([WebConfigurationAuthCallbackPathSchema, WebConfigurationAuthCallbackPathSchema.array()])
        .optional(),
    webhooks_path: zod.preprocess(ensurePathStartsWithSlash, zod.string()).optional(),
    port: zod.number().max(65536).min(0).optional(),
    commands: zod.object({
        build: zod.string().optional(),
        dev: zod.string(),
    }),
    name: zod.string().optional(),
    hmr_server: zod.object({ http_paths: zod.string().array() }).optional(),
});
const webTypes = zod.enum([WebType.Frontend, WebType.Backend, WebType.Background]).default(WebType.Frontend);
export const WebConfigurationSchema = zod.union([
    baseWebConfigurationSchema.extend({ roles: zod.array(webTypes) }),
    baseWebConfigurationSchema.extend({ type: webTypes }),
]);
export const ProcessedWebConfigurationSchema = baseWebConfigurationSchema.extend({ roles: zod.array(webTypes) });
export class App {
    // eslint-disable-next-line max-params
    constructor(name, idEnvironmentVariableName, directory, packageManager, configuration, configurationPath, nodeDependencies, webs, extensions, usesWorkspaces, dotenv, errors) {
        this.name = name;
        this.idEnvironmentVariableName = idEnvironmentVariableName;
        this.directory = directory;
        this.packageManager = packageManager;
        this.configuration = configuration;
        this.configurationPath = configurationPath;
        this.nodeDependencies = nodeDependencies;
        this.webs = webs;
        this.dotenv = dotenv;
        this.allExtensions = extensions;
        this.errors = errors;
        this.usesWorkspaces = usesWorkspaces;
    }
    async updateDependencies() {
        const nodeDependencies = await getDependencies(joinPath(this.directory, 'package.json'));
        this.nodeDependencies = nodeDependencies;
    }
    hasExtensions() {
        return this.allExtensions.length > 0;
    }
    extensionsForType(specification) {
        return this.allExtensions.filter((extension) => extension.type === specification.identifier || extension.type === specification.externalIdentifier);
    }
}
export class EmptyApp extends App {
    constructor() {
        const configuration = { scopes: '', extension_directories: [] };
        super('', '', '', 'npm', configuration, '', {}, [], [], false);
    }
}
/**
 * Given a UI extension and the app it belongs to, it returns the version of the renderer package.
 * Looks for `/node_modules/@shopify/{renderer-package-name}/package.json` to find the real version used.
 * @param uiExtensionType - UI extension whose renderer version will be obtained.
 * @param app - App object containing the extension.
 * @returns The version if the dependency exists.
 */
export async function getUIExtensionRendererVersion(extension) {
    // Look for the vanilla JS version of the dependency (the react one depends on it, will always be present)
    const rendererDependency = extension.dependency;
    if (!rendererDependency)
        return undefined;
    return getDependencyVersion(rendererDependency, extension.directory);
}
export async function getDependencyVersion(dependency, directory) {
    // Split the dependency name to avoid using "/" in windows. Only look for non react dependencies.
    const dependencyName = dependency.replace('-react', '').split('/');
    const pattern = joinPath('node_modules', dependencyName[0], dependencyName[1], 'package.json');
    let packagePath = await findPathUp(pattern, {
        cwd: directory,
        type: 'file',
        allowSymlinks: true,
    });
    if (!packagePath)
        return 'not_found';
    packagePath = await fileRealPath(packagePath);
    // Load the package.json and extract the version
    const packageContent = await readAndParsePackageJson(packagePath);
    if (!packageContent.version)
        return 'not_found';
    return { name: dependency, version: packageContent.version };
}
//# sourceMappingURL=app.js.map