import React from 'react';
import type { PropsWithChildren } from 'react';
import type { AppConfigV2, DispatchActionHook } from '@shopify/app-bridge/client';
import type { History } from '../ClientRouter';
import type { LocationOrHref } from '../RoutePropagator';
/**
 * Provider props
 *
 * @public
 */
export interface Props {
    /** Application configuration */
    config: AppConfigV2;
    /** The child elements to render. */
    children?: React.ReactNode;
    /**
     * Option to set up client routing and route propagation automatically.
     * Passing in a router will allow you to bypass setting these
     * utilities up yourself.
     *
     * If you are using React Router, ensure that the Provider is a child
     * of the router component. For example:
     *
     * import {useMemo} from 'react';
     * import {useLocation, useNavigate, BrowserRouter} from 'react-router-dom';
     * import {Provider} from '@shopify/app-bridge-react';
     * import Routes from './Routes';
     *
     * export function App() {
     *   const navigate = useNavigate();
     *   const location = useLocation();
     *   const history = useMemo(
     *     () => ({replace: (path: string) => navigate(path, {replace: true})}),
     *     [navigate],
     *   );
     *   const router = useMemo(
     *     () => ({
     *       location,
     *       history,
     *     }),
     *     [location, navigate],
     *   );
     *   return (
     *     <Provider
     *       apiKey={{
     *         apiKey: 'API key from Shopify Partner Dashboard',
     *         host: host,
     *       }}
     *       router={router}
     *     >
     *       <Routes />
     *     </Provider>
     *   );
     * }
     *
     * export default function AppWrapper() {
     *   return (
     *     <BrowserRouter>
     *       <App />
     *     </BrowserRouter>
     *   );
     * }
     */
    router?: {
        location: LocationOrHref;
        history: History;
    };
}
/**
 * Create an App Bridge client application from a config and pass it as the
 * value to the context provider.
 *
 * @public
 */
declare function Provider({ config, router, children }: PropsWithChildren<Props>): JSX.Element;
/**
 * Augment actions with clientInterface metadata, identifying use of this library
 *
 * @internal
 */
export declare const setClientInterfaceHook: DispatchActionHook;
interface RouterProps {
    location: LocationOrHref;
    history: History;
}
export declare function useRouter(): RouterProps | undefined;
export default Provider;
