'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const GID_TYPE_REGEXP = /^gid:\/\/[\w-]+\/([\w-]+)\//;
const GID_REGEXP = /\/(\w[\w-]*)(?:\?(.*))*$/;
function parseGidType(gid) {
  const matches = GID_TYPE_REGEXP.exec(gid);

  if (matches && matches[1] !== undefined) {
    return matches[1];
  }

  throw new Error(`Invalid gid: ${gid}`);
}
function parseGid(gid) {
  // prepends forward slash to help identify invalid id
  const id = `/${gid}`;
  const matches = GID_REGEXP.exec(id);

  if (matches && matches[1] !== undefined) {
    return matches[1];
  }

  throw new Error(`Invalid gid: ${gid}`);
}
function parseGidWithParams(gid) {
  // prepends forward slash to help identify invalid id
  const id = `/${gid}`;
  const matches = GID_REGEXP.exec(id);

  if (matches && matches[1] !== undefined) {
    const params = matches[2] === undefined ? {} : fromEntries(new URLSearchParams(matches[2]).entries());
    return {
      id: matches[1],
      params
    };
  }

  throw new Error(`Invalid gid: ${gid}`);
}
function composeGidFactory(namespace) {
  return function composeGid(key, id, params = {}) {
    const gid = `gid://${namespace}/${key}/${id}`;
    const paramKeys = Object.keys(params);

    if (paramKeys.length === 0) {
      return gid;
    }

    const paramString = new URLSearchParams(params).toString();
    return `${gid}?${paramString}`;
  };
}
const composeGid = composeGidFactory('shopify');
function isGidFactory(namespace) {
  return function isGid(gid, key) {
    if (!gid.startsWith(`gid://${namespace}/`)) {
      return false;
    }

    try {
      if (key !== undefined && parseGidType(gid) !== key) {
        return false;
      }
    } catch {
      return false;
    } // prepends forward slash to help identify invalid id


    const id = `/${gid}`;
    return GID_REGEXP.test(id);
  };
}
const isGid = isGidFactory('shopify');
function nodesFromEdges(edges) {
  return edges.map(({
    node
  }) => node);
}
function keyFromEdges(edges, key) {
  return edges.map(({
    node
  }) => node[key]);
} // Once we update to Node 12, we can drop this helper to use `Object.fromEntries` instead.

function fromEntries(entries) {
  const obj = {};

  for (const [key, val] of entries) {
    obj[key] = val;
  }

  return obj;
}

exports.composeGid = composeGid;
exports.composeGidFactory = composeGidFactory;
exports.isGid = isGid;
exports.isGidFactory = isGidFactory;
exports.keyFromEdges = keyFromEdges;
exports.nodesFromEdges = nodesFromEdges;
exports.parseGid = parseGid;
exports.parseGidType = parseGidType;
exports.parseGidWithParams = parseGidWithParams;
